package pluginloader.gradle

import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.publish.maven.MavenPublication
import org.gradle.jvm.tasks.Jar
import java.nio.file.Files


class GradlePlugin: Plugin<Project> {
    override fun apply(project: Project) {
        //Apply plugins
        project.applyPlugin("org.jetbrains.kotlin.jvm")
        project.applyPlugin("org.jetbrains.kotlin.plugin.serialization")
        //Used on publish
        project.applyPlugin("maven-publish")
        //Used on publish to maven central
        project.applyPlugin("signing")

        //Maven central used anyway, because api jars there
        project.repositories.add(project.repositories.mavenCentral())

        //Some libs bundled with pluginloader jar
        project.compileDependency("org.jetbrains.kotlin:kotlin-stdlib:$kotlinVersion")
        project.compileDependency("org.jetbrains.kotlinx:kotlinx-serialization-core:$kotlinSerializationVersion")
        project.compileDependency("org.jetbrains.kotlinx:kotlinx-serialization-json:$kotlinSerializationVersion")
        project.compileDependency("io.github.pluginloader:api:$pluVersion")

        //Special configuration for
        val depConf = project.createConfiguration("dependency")
        val depMavenConf = project.createConfiguration("mavenDependency")

        //I don't find any way better to get plugin
        val publishing = (project as org.gradle.api.plugins.ExtensionAware)
            .extensions.getByName("publishing") as org.gradle.api.publish.PublishingExtension

        //Generate maven publishing data by default
        publishing.publications.create("java", MavenPublication::class.java){
            it.from(project.components.getByName("java"))
        }

        //Add api to gradle
        val config = Config(project)
        project.extensions.add("plu", config)

        //Write information about dependencies
        project.tasks.getByPath("jar").apply jar@ {
            this as Jar
            archiveFileName.set("${project.name}.jar")

            doFirst{
                //Delete older builds data
                project.file(hackyPath).apply{if(exists()) deleteRecursively()}

                //Create files for dependency plugins
                project.addToJarPerDependency(depConf, writeToFile = {
                    "$group:$name:$version"
                }){"$name.dependency"}

                //For mavenDependency plugins
                project.addToJarPerDependency(depMavenConf){"$group;$name;$version.mavenDependency"}

                //Mark expand
                config.expand?.apply{project.addFileToJar("$this.expand")}
            }
        }

        //Auto update project name & package, very useful for templates
        project.file("plu_project.txt").apply update@ {
            val path = toPath()
            //'Files' functions work more stable on Windows
            if(Files.notExists(path))return@update

            //Old project name in plu_project.txt, expect something like 'template'
            val oldName = readText().split("\n")[0]
            //New name, expect new project name
            val newName = project.file(".").name

            //I hope, if name contains 'template', is mean this is template
            if(newName.contains("template"))return@update

            //Remove template mark
            Files.delete(path)

            //Replace package in kotlin
            val src = project.file("src/main/kotlin/Plugin.kt")
            val srcPath = src.toPath()
            if(Files.exists(srcPath)) src.writeText(src.readText().replace(oldName, newName.replace("-", "_")))

            //Replace project name in gradle
            val settings = project.file("./settings.gradle.kts")
            val settingsPath = settings.toPath()
            if(Files.exists(settingsPath)) settings.writeText(settings.readText().replace(oldName, newName))
        }
    }
}