/*

 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.github.shuttlelang;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Shuttle {

	private static final Pattern REGEX_AUTOLINKS_URL = Pattern.compile("(?m)(?<!\\()((https?|ftp)://[\\w./&;]*)\\b(?=\\s|\\z)");
	private static final Pattern REGEX_AUTOLINKS_EMAIL = Pattern.compile("\\b([-.\\w]+\\@[-a-z0-9]+(\\.[-a-z0-9]+)*\\.[a-z]+)\\b(?=\\s|\\z)");
	private static final Pattern REGEX_BLOCKQUOTES = Pattern.compile("(?m)^>[ \t]*((?s:.+?))(?=\\n[^>]|\\n{2}|\\z)");
	private static final Pattern REGEX_BLOCKQUOTES_REMOVE_MARK = Pattern.compile("(?m)^[ \\t]*>[ \\t]*");
	private static final Pattern REGEX_CODE_SPANS = Pattern.compile("(?<!\\\\)(`+)(.+?)(?<!`)\\1(?!`)");
	private static final Pattern REGEX_FENCED_CODE_BLOCKS = Pattern.compile("(?m)^`{3}(.*)\n((?s:.+?))`{3}");
	private static final Pattern REGEX_ENCODE_AMPERSAND = Pattern.compile("&(?!\\#?[xX]?(?:[0-9a-fA-F]+|\\w+);)");
	private static final Pattern REGEX_ENCODE_DOLLAR = Pattern.compile("\\$");
	private static final Pattern REGEX_ENCODE_GREATER_THEN = Pattern.compile("(?<![a-z/?\"\\$!-])>");
	private static final Pattern REGEX_ENCODE_LESS_THEN = Pattern.compile("<(?![a-z/?\\$!])");
	private static final Pattern REGEX_HASH_HTML_COMMENTS = Pattern.compile("(?m)^<!--(?s:.+?)-->");
	private static final Pattern REGEX_HASH_HTML_BLOCKS = Pattern.compile("(?mi)^[ ]*<(div|table|a|blockquote|code|d[ltd]|h[1-6]|ol|p|pre|ul|kbd)\\b(?s:.+?)</\\1>(?=\\n{2,}|\\Z)");
	private static final Pattern REGEX_HASH_HTML_EMPTY_TAGS = Pattern.compile("(?mi)^<(br|hr|img).*>");
	private static final Pattern REGEX_HASH_HTML_NESTED_BLOCKS = Pattern.compile("(?mi)^<(div|table|a|blockquote|code|d[ltd]|h[1-6]|ol|p|pre|ul|kbd)\\b(.+\n)+\n[ ]{4}(?s:.+?)^</\\1>");
	private static final Pattern REGEX_HEADERS = Pattern.compile("(?m)^[ \\t]*(\\#{1,6})[ \\t]*(.+?)[ \\t]*$");
	private static final Pattern REGEX_HORIZONTAL_RULES = Pattern.compile("(?m)^[ \\t]{0,3}(-[ ]?){3,}[ \\t]*$");
	private static final Pattern REGEX_IMAGES_INLINE = Pattern.compile("!\\[(.+)\\][ ]?\\((.+)\\)");
	private static final Pattern REGEX_IMAGES_INLINE_WITH_TITLE = Pattern.compile("!\\[(.+)\\][ ]?\\((\\S+)[ ]*\"(.*)\"[ ]*\\)");
	private static final Pattern REGEX_IMAGES_REFERENCED = Pattern.compile("[!]\\[(.*?)\\][ ]?(?:\\n[ ]*)?\\[(.*?)\\]");
	private static final Pattern REGEX_INDENT = Pattern.compile("(?m)^");
	private static final Pattern REGEX_LINK_DEFINITIONS = Pattern.compile("(?m)^[ \t]*\\[([^\\[]+)\\]:[ \\t]*\\n?[ \\t]*<?[\\[]?(\\S+?)[\\]]?>?[ \\t]*\\n?[ \\t]*(?:[\"(](.+?)[\")][ \\t]*)?(?:\\n+|\\Z)");
	private static final Pattern REGEX_LINKS_REFERENCED = Pattern.compile("\\[([^\\[]*?)\\][ ]?(?:\\n[ ]*)?\\[(.*?)\\]");		
	private static final Pattern REGEX_LINKS_INLINE = Pattern.compile("(\\[(.*?)\\][ ]?\\([ \\t]*<?(.+?)>?[ \\t]*((['\"])(.*?)\\5)?\\))");
	private static final Pattern REGEX_LIST_ITEMS = Pattern.compile("(?m)^[ \t]{0,3}(?:-|\\d+[.])([ \t]+((?s:.+?)(?=^[ \t]{0,3}-|^[ \t]{0,3}\\d+[.]|\\z))|$\\n)");	private static final Pattern REGEX_ORDERED_LISTS = Pattern.compile("(?m)^[ \t]{0,3}(\\d+[.](?s:.+?)(?=\\n{3}|\\n{2}<ul>|\\n{2}[^(\\d\\s)]|\\n{2}\\d[^.]|\\Z))");
	private static final Pattern REGEX_UNORDERED_LISTS = Pattern.compile("(?m)^[ \t]{0,3}(-[ \t]+(?s:.+?)(?=\\n{3}|\\n{2}[^-\\s]|\\n+\\d[.]|\\Z))");
	private static final Pattern REGEX_TABLES_REMOVE_MARK = Pattern.compile("(?m)^[|]");
	private static final Pattern REGEX_TABLES_WITH_LEADING_PIPE = Pattern.compile("(?m)^[ ]{0,3}[|](.+)\n[ ]{0,3}[|]([ ]*[-:]+[-| :]*)\n((?>[ ]*[|].*\n)*)(?=\\n|\\Z)");
	private static final Pattern REGEX_TABLES_WITHOUT_LEADING_PIPE = Pattern.compile("(?m)^[ ]{0,3}(\\S.*[|].*)\n[ ]{0,3}([-:]+[ ]*[|][-| :]*)\n((?>.*[|].*\n)*)(?=\\n|\\Z)");
	
	public interface Renderer {
		String renderBlockquote(String text, boolean nested);
		String renderCodeSpan(String text);
		String renderDel(String text);
		String renderEm(String text);
		String renderFencedCodeBlocks(String code, String lang);
		String renderHeader(String text, int level);
		String renderHtml(String html, boolean nested, int i);
		String renderHr();
		String renderImage(String src, String title, String alt);
		String renderLineBreak();
		String renderLink(String url, String title, String text);
		String renderList(String body, boolean nested, boolean ordered);
		String renderListItem(String text);
		String renderParagraph(String text);
		String renderStrong(String text);
		String renderTable(String head, String body);
		String renderTableRow(String text);
		String renderTableCell(String text, boolean header, String align);
	}
	
    private interface Replacement {
    	String replace(Matcher m);
    }
    
    public enum Extension { AUTOLINKS, TABLES }
	private List<Extension> extensions;
	private Map<String, String> linkUrls = new HashMap<>();
	private Map<String, String> linkTitles = new HashMap<>();
	private Map<String, String> htmlHash = new HashMap<>();
	private Map<String, String> inlineHash = new HashMap<>();
	private Renderer renderer;
	
	public Shuttle(Extension... extensions) {
		this.extensions = Arrays.asList(extensions);
	}
	
	public String parse(String text) {
		return parse(text, new Html5Renderer());
	}
		
	public String parse(String text, Renderer renderer) {	
		this.renderer = renderer;
		linkUrls.clear();
		linkTitles.clear();
		htmlHash.clear();
		inlineHash.clear();
		
		text = text.replaceAll("\r\n|\r", "\n");
		text = text.replaceAll("\\t", "    ");
		text = text.replaceAll("(?m)[ \t]+$", "");
		text = doFencedCodeBlocks(text);
		text = hashHTMLBlocks(text);
		text = stripLinkDefinitions(text);
		text = doBlockElements(text, false);
		text = unEscapeSpecialChars(text);
		text = text.replaceAll("(?m)^[ \t]*$", "");
		return text;
	}
	
	private String doFencedCodeBlocks(String text) {
		return replaceAll(text, REGEX_FENCED_CODE_BLOCKS, new Replacement() {
			public String replace(Matcher m) {
				return renderer.renderFencedCodeBlocks(encodeCode(m.group(2).trim()), m.group(1));
			}
		});
	}
	
	private String hashHTMLBlocks(String text) {
		Replacement protectHTML = new Replacement() {
            public String replace(Matcher m) {
                String literal = m.group();
                return "\n" + hashHtml(literal) + "\n\n";
            }
		};
		text = replaceAll(text, REGEX_HASH_HTML_COMMENTS, protectHTML);
		text = replaceAll(text, REGEX_HASH_HTML_NESTED_BLOCKS, protectHTML);
		text = replaceAll(text, REGEX_HASH_HTML_BLOCKS, protectHTML);	
		return replaceAll(text, REGEX_HASH_HTML_EMPTY_TAGS, protectHTML);	
	}
	
	private String stripLinkDefinitions(String text) {
		return replaceAll(text, REGEX_LINK_DEFINITIONS, new Replacement() {
			public String replace(Matcher m) {
				String id = m.group(1).toLowerCase();
	            linkUrls.put(id, m.group(2));
	            linkTitles.put(id, m.group(3));
				return "";
			}
		});
	}
	
	private String doBlockElements(String text, boolean nested) {		
		text = doBlockquotes(text, renderer, nested);
		text = doFencedCodeBlocks(text);
		text = REGEX_HORIZONTAL_RULES.matcher(text).replaceAll(renderer.renderHr());
		text = doUnorderedLists(text, nested);
		text = doOrderedLists(text, nested);
		text = doHeaders(text);
		text = doTables(text);
		text = hashHTMLBlocks(text);
		return formParagraphs(text, nested);
		//return text;
	}
	
	private String doBlockquotes(String text, final Renderer renderer, final boolean nested) {
		return replaceAll(text, REGEX_BLOCKQUOTES, new Replacement() {
			public String replace(Matcher m) {
				String body = doBlockElements(REGEX_BLOCKQUOTES_REMOVE_MARK.matcher(m.group()).replaceAll(""), true);
				return renderer.renderBlockquote(body, nested);
			}
		});
	}
	
	private String doUnorderedLists(String text, final boolean nested) {
		return replaceAll(text, REGEX_UNORDERED_LISTS, new Replacement() {
			public String replace(Matcher m) {
				return renderer.renderList(doListItems(m.group(1)), nested, false);
			}
		});
	}
	
	private String doOrderedLists(String text, final boolean nested) {
		return replaceAll(text, REGEX_ORDERED_LISTS, new Replacement() {
			public String replace(Matcher m) {
				return renderer.renderList(doListItems(m.group(1)), nested, true);
			}
		});
	}
	
	private String doListItems(String list) {
		return replaceAll(list, REGEX_LIST_ITEMS, new Replacement() {
			public String replace(Matcher m) {
				return renderer.renderListItem(doBlockElements(m.group(1).trim().replaceAll("(?m)^[ \\t]{0,4}", ""), true));
			}
		});
	}
	
	private String doHeaders(String text) {
		return replaceAll(text, REGEX_HEADERS, new Replacement() {
			public String replace(Matcher m) {
	            return renderer.renderHeader(doSpanElements(m.group(2).trim()), m.group(1).length());
			}
		});
	}
	
	private String doTables(String text) {
		if(extensions.contains(Extension.TABLES)) {
			Replacement tables = new Replacement() {
				public String replace(Matcher m) {
					return doTable(m.group(), m.group(1), m.group(2), REGEX_TABLES_REMOVE_MARK.matcher(m.group(3)).replaceAll(""));
				}
			};
			text = encodeEscapes(text, new char[]{'|'}, "\\\\\\");
			text = replaceAll(text, REGEX_TABLES_WITH_LEADING_PIPE, tables);
			text = replaceAll(text, REGEX_TABLES_WITHOUT_LEADING_PIPE, tables);
		}
		return text;
		
	}
	
	private String doTable(String table, String head, String underline, String content) {
		String[] headers = head.split("[|]");
		String[] cols = underline.split("[|]");
		String[] attributes = doTableAttributes(cols);
		String[] rows = content.split("\n");
		return renderer.renderTable(doTableHead(attributes, headers), doTableBody(attributes, rows));
	}
	
	private String[] doTableAttributes(String[] cols) {
		String[] attributes = new String[cols.length];
		for(int i=0;i<cols.length;i++) {
			String alignment = cols[i].trim();
			if(alignment.matches("^-+:$")) {
				attributes[i] = "right";
			} else if(alignment.matches("^:-+:$")) {
				attributes[i] = "center";
			} else if(alignment.matches("^:-+$")) {
				attributes[i] = "left";
			} else {
				attributes[i] = "";
			}
		}
		return attributes;
	}
	
	private String doTableHead(String[] attributes, String[] headers) {
		StringBuilder th = new StringBuilder();
		for(int i=0;i<headers.length;i++) {
			th.append(renderer.renderTableCell(doSpanElements(headers[i]).trim(), true, attributes[i]));
		}
		return renderer.renderTableRow(th.toString());
	}
	
	private String doTableBody(String[] attributes, String[] rows) {
		StringBuilder output = new StringBuilder();
		for(int i=0; i < rows.length; i++) {
			String[] cells = rows[i].split("[|]");
			StringBuilder td = new StringBuilder();
			for(int j=0;j<cells.length;j++) {
				String text = doSpanElements(cells[j]).trim();
				if(attributes.length > j) {
					td.append(renderer.renderTableCell(text, false, attributes[j]));
				} else {
					td.append(renderer.renderTableCell(text, false, ""));
				}
			}
			output.append(renderer.renderTableRow(td.toString()));
		}
		return output.toString();
	}
	
	private String formParagraphs(String text, boolean nested) {
		String[] paragraphs = text.trim().split("\\n{2,}");
		for(int i=0; i < paragraphs.length; i++) {
			String p = paragraphs[i].trim();
			if (htmlHash.containsKey(p)) {
				paragraphs[i] = renderer.renderHtml(htmlHash.get(p), nested, i);
			} else  {
				p = doSpanElements(p).replaceAll("\\n", renderer.renderLineBreak());
				if(i == 0 && nested) {
					paragraphs[i] = p;
					if(paragraphs.length == 1) {
						return p;
					}
				} else if(!"".equals(p)){
					paragraphs[i] = renderer.renderParagraph(p);
				}
			}
			if(i > 0 && nested) {
				paragraphs[i] = REGEX_INDENT.matcher(paragraphs[i]).replaceAll("    ");
			}
		}
		return join(paragraphs, "\n\n") + "\n";
	}
	
	private String doSpanElements(String text) {
		text = doCodeSpans(text);
		text = encodeBackslashEscapes(text);
		text = doImages(text);
		text = doLinks(text);
		text = doAutoLinks(text);
		text = doFormatting(text);
		text = encodeAmpsAndAngles(text);
		return text;
	}
	
	private String doCodeSpans(String text) {
		text = replaceAll(text, REGEX_CODE_SPANS, new Replacement() {
            public String replace(Matcher m) {
            	return renderer.renderCodeSpan(encodeCode(m.group(2).trim()));
            }
        });
		return replaceAll(text, Pattern.compile("(?m)<code>.*</code>"), new Replacement() {
			public String replace(Matcher m) {
				String code = m.group().replaceAll("\\\\", "\\\\\\\\");
				return hashInline(code);
			}
		});
	}
	
	private String encodeBackslashEscapes(String text) {
		char[] escapedChars = "[]()#*~_-.`!".toCharArray();
		text = text.replaceAll("(?m)\\\\\\\\", hashInline("\\"));	
		text = encodeEscapes(text, escapedChars, "\\\\\\");
		return text;
	}
	
	private String encodeEscapes(String text, char[] chars, String slashes) {
        for (final char ch : chars) {
            Pattern regex = Pattern.compile(slashes + ch, Pattern.MULTILINE);
            text = replaceAll(text, regex, new Replacement() {
				public String replace(Matcher m) {
					return hashInline(String.valueOf(ch));
				}
			});
        }
        return text;
    }
	
	private String doImages(String text) {	
		text = REGEX_IMAGES_INLINE_WITH_TITLE.matcher(text).replaceAll(renderer.renderImage("$2", "$3", "$1"));
		text = replaceAll(text, REGEX_IMAGES_INLINE, new Replacement() {
			public String replace(Matcher m) {
				return renderer.renderImage(m.group(2), null, hashInline(m.group(1)));
			}
		});
    	return replaceAll(text, REGEX_IMAGES_REFERENCED, new Replacement() {
			public String replace(Matcher m) {
            	String alt = m.group(1);
            	String id = m.group(2).toLowerCase();
            	if ("".equals(id)) {
                	id = alt.toLowerCase();
            	}
            	if (linkUrls.containsKey(id)) {
            		return renderer.renderImage(linkUrls.get(id), linkTitles.get(id), alt);
            		
            	}
            	return m.group();
			}
		});
	}
	
	private String doLinks(String text) {
		text = replaceAll(text, REGEX_LINKS_REFERENCED, new Replacement() {
			public String replace(Matcher m) {
                String linkText = m.group(1);
                String id = m.group(2).toLowerCase();
                if ("".equals(id)) { 
                    id = linkText.toLowerCase();
                }
                if(linkUrls.containsKey(id)) {
                	return renderer.renderLink(linkUrls.get(id), linkTitles.get(id), linkText);
                }
				return m.group();
			}
		});
		return replaceAll(text, REGEX_LINKS_INLINE, new Replacement() {
			public String replace(Matcher m) {
				return renderer.renderLink(m.group(3), m.group(6), m.group(2));
			}
		});
	}
	
	private String doAutoLinks(String text) {
		if(extensions.contains(Extension.AUTOLINKS)) {
			text = REGEX_AUTOLINKS_URL.matcher(text).replaceAll(renderer.renderLink("$1", "", "$1"));
			text = REGEX_AUTOLINKS_EMAIL.matcher(text).replaceAll(renderer.renderLink("mailto:$1", "", "$1"));
		}
		return text;
	}
	
	private String doFormatting(String text) {
		text = text.replaceAll("(?<=^|\\W)\\_(.+?)\\_(?=\\W|$)", renderer.renderEm("$1"));
		text = text.replaceAll("(?<=^|\\W)\\*(.+?)\\*(?=\\W|$)", renderer.renderStrong("$1"));
		return text.replaceAll("(?<=^|\\W)\\~(.+?)\\~(?=\\W|$)", renderer.renderDel("$1"));
	}
	
	private String encodeAmpsAndAngles(String text) {
		text = REGEX_ENCODE_AMPERSAND.matcher(text).replaceAll("&amp;");
		text = REGEX_ENCODE_LESS_THEN.matcher(text).replaceAll("&lt;");
		text = REGEX_ENCODE_GREATER_THEN.matcher(text).replaceAll("&gt;");
		return REGEX_ENCODE_DOLLAR.matcher(text).replaceAll("&#036;");
	}
		
	private String encodeCode(String text) {
		text = encodeEscapes(text, new char[]{'`'}, "\\\\\\");
		text = text.replaceAll("\\\\", "\\\\\\\\");
		text = text.replaceAll("<", "&lt;");
		text = text.replaceAll(">", "&gt;");
		text = text.replaceAll("\\$", "&#036");
		return text;
	}
	
	private String hashHtml(String literal) {
        String hash = "===SHUTTLE_HTML~" + htmlHash.size() + "===";
        htmlHash.put(hash, literal);
        return hash;
	}
	
	private String hashInline(String literal) {
        String hash = "===SHUTTLE_INLINE~" + inlineHash.size() + "===";
        inlineHash.put(hash, literal);
        return hash;
	}

	private String join(final String[] array, String separator) {
        StringBuilder buf = new StringBuilder(array.length * 16);
        for (int i = 0; i < array.length; i++) {
            if (i > 0) {
                buf.append(separator);
            }
            if (array[i] != null) {
                buf.append(array[i]);
            }
        }
        return buf.toString();
    }
	
	private String replaceAll(String text, Pattern p, Replacement replacement) {
        Matcher m = p.matcher(text);
        StringBuffer sb = new StringBuffer();
        while (m.find()) {
            m.appendReplacement(sb, replacement.replace(m));
        }
        m.appendTail(sb);
        return sb.toString();
	}
	
	private String unEscapeSpecialChars(String text) {
        for (String hash : inlineHash.keySet()) {
            String plaintext = inlineHash.get(hash);
            if(plaintext == "\\") {
            	text = text.replaceAll(hash, "\\\\");
            } else {
            	text = text.replaceAll(hash, plaintext);
            }
        }
        return text;
    }
	
	class Html5Renderer implements Renderer  {

		public String renderBlockquote(String body, boolean nested) {
			StringBuilder out = new StringBuilder();
			if(nested) {
				out.append("\n\n");
			}
			return out.append("<blockquote>").append(body).append("</blockquote>\n\n").toString();
		}
		
		public String renderCodeSpan(String text) {
			return "<code>" + text + "</code>";
		}
		
		public String renderDel(String text) {
			return "<del>" + text + "</del>";
		}

		public String renderEm(String text) {
			return "<em>" + text + "</em>";
		}
		
		public String renderFencedCodeBlocks(String code, String lang) {
			StringBuilder block = new StringBuilder("<pre><code");
			if(lang.length() > 0) {
				block.append(" class=\"lang-").append(lang).append("\"");
			}
			return block.append(">").append(code).append("</code></pre>\n").toString();
		}
		
		public String renderHeader(String text, int level) {
			 String tag = "h" + level;
	         return "<" + tag + ">" + text + "</" + tag + ">\n\n";
		}
		
		public String renderHr() {
			return "<hr>\n";
		}
		
		public String renderHtml(String html, boolean nested, int i) {
			if(i == 0 && nested) {
				return "\n" + REGEX_INDENT.matcher(html).replaceAll("    ");
			} else {
				return html;
			}
		}
		
		public String renderImage(String src, String title, String alt) {
			StringBuilder image = new StringBuilder("<img src=\"");
			image.append(src).append("\" alt=\"").append(alt).append("\"");
			if (title != null && !title.equals("")) {
        		image.append(" title=\"").append(title).append("\"");
        	}
			return image.append(">").toString();
		}
		
		public String renderLineBreak() {
			return "<br>\n";
		}
		
		public String renderLink(String url, String title, String text) {
        	StringBuilder link = new StringBuilder("<a href=\"" + url.replaceAll(" ", "%20") + "\"");
        	if (title != null && !title.equals("")) {
        		link.append(" title=\"" + title + "\"");
        	}
        	return link.append(">").append(text).append("</a>").toString();
		}
		
		public String renderList(String body, boolean nested, boolean ordered) {
			String tag = ordered ? "ol" : "ul";
			String list = "\n<" + tag + ">\n" + REGEX_INDENT.matcher(body).replaceAll("    ") + "</" + tag + ">\n";
			return nested ? list : list + "\n";
		}
		
		public String renderListItem(String text) {
			return "<li>" + text + "</li>\n";
		}
		
		public String renderParagraph(String body) {
			return "<p>" + body + "</p>";
		}
		
		public String renderStrong(String text) {
			return "<strong>" + text + "</strong>";
		}

		public String renderTable(String head, String body) {
			StringBuilder table = new StringBuilder("<table>\n    <thead>\n");
			table.append(head).append("    </thead>\n    <tbody>\n");
			return table.append(body).append("    </tbody>\n</table>\n").toString();
		}
		
		public String renderTableRow(String text) {
			return "        <tr>\n" + REGEX_INDENT.matcher(text).replaceAll("            ") + "        </tr>\n";
		}

		public String renderTableCell(String text, boolean header, String align) {
			String tag = header ? "th" : "td";
			if(align != "") {
				return "<" + tag + " style=\"text-align: " + align + ";\">" + text + "</" + tag + ">\n";
			}
			return "<" + tag + ">" + text + "</" + tag + ">\n";
		}
	}
}