package com.tidal.utils.junit;

import com.tidal.utils.filehandlers.FileOutWriter;
import com.tidal.utils.filehandlers.FilePaths;
import com.tidal.utils.filehandlers.FileReader;
import com.tidal.utils.propertieshandler.Config;
import com.tidal.utils.utils.Helper;
import com.tidal.utils.xml.Processors;
import com.tidal.utils.xml.XMLReader;
import com.tidal.utils.xml.XmlBuilder;
import org.jetbrains.annotations.NotNull;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.List;
import java.util.Locale;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import static com.tidal.utils.junit.Template.resultXMLTemplate;

public class ResultParser {

    private static final Logger logger = Logger.getLogger("ResultParser");

    //Even if there is no rerun the output file name will be 'TestResultFinal.xml'
    private static final String FINAL_RESULT_FILE = "TestResultFinal.xml";
    private static final String CURRENT_RUNNER_FILE = Config.RUNNER + ".xml";
    private static final Path TARGET_FOLDER_PATH = Paths.get(Helper.getAbsoluteFromRelativePath(FilePaths.TARGET_FOLDER_PATH.getPath()));
    private static final Path PATH_TO_WRITE_FILE = Paths.get(TARGET_FOLDER_PATH.toString(), "junit-reports", FINAL_RESULT_FILE);
    private static final String XML_DECLARATION = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n";

    private ResultParser() {
    }


    /**
     * Method to parse the Junit xml file generated by the TestNgRunner.
     * Failed test cases are filtered out as they will be tried again in the next run.
     */
    public static void parseRunnerResult() {
        logger.info("Parsing Result from first run");

        String xmlToWrite = getCurrentRunnerResultXml();

        if (Config.RETRY_FAILED_TESTS) {
            List<Node> passedTests = getPassedTestCases();
            xmlToWrite = XML_DECLARATION + filterFailedTestCases(passedTests);
        }
        FileOutWriter.writeFileTo(xmlToWrite, PATH_TO_WRITE_FILE.toString());
    }

    /**
     * Method to merge the runner and re-runner xml files.
     * <p>Existing information from the runner result contained in FINAL_RESULT_FILE will be copied over to the new file</p>
     */
    public static void parseReRunnerResult() {
        logger.info("Parsing Result from retry run");

        List<Node> runnerResult = getTestCasesFrom(FINAL_RESULT_FILE);
        List<Node> reRunnerResult = getTestCasesFrom(CURRENT_RUNNER_FILE);

        String mergedXML = mergeResult(runnerResult, reRunnerResult);

        //Compute failure count, total time and number of tests from both the run files
        String failureCount = XMLReader.getValue("testsuite.@failures", getCurrentRunnerResultXml());
        logger.info(String.format("Combined failure count %s", failureCount));

        String totalTime = String.format("%.3f", getTotalRuntime(runnerResult, reRunnerResult));
        logger.info(String.format("Total run time from first and second run %s", totalTime));

        String totalTests = String.valueOf(runnerResult.size() + reRunnerResult.size());
        logger.info(String.format("Total tests run: %s", totalTests));

        //update merged xml with new values
        mergedXML = new XmlBuilder("testsuite", mergedXML)
                .withUpdatedAttribute("time", totalTime)
                .withUpdatedAttribute("failures", failureCount)
                .withUpdatedAttribute("tests", totalTests)
                .extract();

        FileOutWriter.writeFileTo(XML_DECLARATION + mergedXML, PATH_TO_WRITE_FILE.toString());
    }

    private static float getTotalRuntime(List<Node> runnerResult, List<Node> reRunnerResult) {
        float totalTestRunTime = 0;
        float runTime = 0;

        NumberFormat numberFormat = NumberFormat.getNumberInstance(Locale.ROOT);

        for (Node n : runnerResult) {
            try {
                runTime = numberFormat.parse(n.getAttributes().getNamedItem("time").getTextContent()).floatValue();
                totalTestRunTime = totalTestRunTime + runTime;
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }
        for (Node n : reRunnerResult) {
            try {
                runTime = numberFormat.parse(n.getAttributes().getNamedItem("time").getTextContent()).floatValue();
                totalTestRunTime = totalTestRunTime + runTime;
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }
        logger.info(String.format("Total run time is %f", totalTestRunTime));
        return totalTestRunTime;
    }

    private static String mergeResult(List<Node> runnerResult, List<Node> reRunnerResult) {
        Document testResult = new XmlBuilder(resultXMLTemplate()).getDocument();
        Element element = testResult.getDocumentElement();

        for (Node n : runnerResult) {
            Node newNode = element.getOwnerDocument().importNode(n, true);
            element.appendChild(newNode);
        }

        for (Node n : reRunnerResult) {
            Node newNode = element.getOwnerDocument().importNode(n, true);
            element.appendChild(newNode);
        }
        return Processors.transformer(testResult);
    }

    private static String filterFailedTestCases(List<Node> passedTests) {
        Document testResultTemplate = new XmlBuilder(resultXMLTemplate()).getDocument();
        Element element = testResultTemplate.getDocumentElement();

        for (Node n : passedTests) {
            Node newNode = element.getOwnerDocument().importNode(n, true);
            element.appendChild(newNode);
        }
        return Processors.transformer(testResultTemplate);
    }

    public static List<Node> getTestCasesFrom(String fileName) {
        String xmlResult = FileReader.readFileToString(fileName, TARGET_FOLDER_PATH);
        return XMLReader.getNodes("testsuite.testcase", xmlResult);
    }

    private static List<Node> getPassedTestCases() {
        String xmlResult = FileReader.readFileToString(CURRENT_RUNNER_FILE, TARGET_FOLDER_PATH);

        List<Node> testCasesList = XMLReader.getNodes("testsuite.testcase", xmlResult);

        final List<Node> passedTests = testCasesList.stream()
                .filter(e -> {
                    if (e == null) return false;
                    Element element = (Element) e;
                    return element.getElementsByTagName("failure").getLength() == 0;
                })
                .collect(Collectors.toList());

        logger.info(String.format("Passed test count is %d", passedTests.size()));
        return passedTests;
    }

    /**
     * This is the output from the current runner file.
     *
     * @return parsed xml
     */
    private static @NotNull String getCurrentRunnerResultXml() {
        return FileReader.readFileToString(CURRENT_RUNNER_FILE, TARGET_FOLDER_PATH);
    }

}
