/*
 * Copyright (c) 2024-2025 Tore Eide Andersen
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.github.torand.openapi2java.writers.java;

import io.github.torand.openapi2java.generators.Options;
import io.github.torand.openapi2java.model.EnumInfo;
import io.github.torand.openapi2java.writers.BaseWriter;
import io.github.torand.openapi2java.writers.EnumWriter;

import java.io.Writer;

import static io.github.torand.javacommons.collection.CollectionHelper.nonEmpty;
import static io.github.torand.openapi2java.utils.StringUtils.joinCsv;

/**
 * Writes Java code for an enum.
 */
public class JavaEnumWriter extends BaseWriter implements EnumWriter {

    public JavaEnumWriter(Writer writer, Options opts) {
        super(writer, opts);
    }

    @Override
    public void write(EnumInfo enumInfo) {
        writeLine("package %s;", opts.getModelPackage(enumInfo.modelSubpackage()));
        writeNewLine();

        if (nonEmpty(enumInfo.aggregatedNormalImports())) {
            enumInfo.aggregatedNormalImports().forEach(i -> writeLine("import %s;".formatted(i)));
            writeNewLine();
        }

        if (nonEmpty(enumInfo.aggregatedStaticImports())) {
            enumInfo.aggregatedStaticImports().forEach(i -> writeLine("import static %s;".formatted(i)));
            writeNewLine();
        }

        enumInfo.annotationsAsStrings().forEach(this::writeLine);

        writeLine("public enum %s {".formatted(enumInfo.name()));
        writeIndent(1);
        writeLine(joinCsv(enumInfo.constants()));
        writeLine("}");
    }
}
