/*
 *  Licensed to the author under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package net.java.quickcheck.generator.iterable;

import java.util.Iterator;
import java.util.NoSuchElementException;

import net.java.quickcheck.Generator;
import net.java.quickcheck.QuickCheck;
import net.java.quickcheck.util.Assert;

public final class Iterables {

	private Iterables() {
	}

	/**
	 * Convert a generator into a {@link Iterable iterable}.<br>
	 * 
	 * The generator will be run {@link QuickCheck#MAX_NUMBER_OF_RUNS} times.
	 */
	public static <T> Iterable<T> toIterable(final Generator<T> generator) {
		return toIterable(generator, QuickCheck.MAX_NUMBER_OF_RUNS);
	}
	
	/**
	 * Convert a generator into a {@link Iterable iterable}.
	 * 
	 * @param numberOfRuns to execute the runner
	 */
	public static <T> Iterable<T> toIterable(final Generator<T> generator, final int numberOfRuns) {
		Assert.notNull(generator, "generator");
		Assert.greaterOrEqual(0.0, numberOfRuns, "number of runs");
		return new Iterable<T>() {

			@Override
			public Iterator<T> iterator() {
				return new Iterator<T>() {

					private int runs;

					@Override
					public boolean hasNext() {
						return runs < numberOfRuns;
					}

					@Override
					public T next() {
						if (!hasNext())
							throw new NoSuchElementException();
						runs++;
						return generator.next();
					}

					@Override
					public void remove() {
						throw new UnsupportedOperationException();
					}
				};
			}
		};
	}
}