/*
 *  Licensed to the author under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package net.java.quickcheck.generator.support;

import static java.lang.Math.*;
import static java.lang.String.*;

import java.util.Calendar;
import java.util.Date;
import java.util.concurrent.TimeUnit;

import net.java.quickcheck.Generator;

public class DateGenerator implements Generator<Date> {

	private VetoableGenerator<Long> generator;

	public DateGenerator(TimeUnit precision, long low, long high, int tries) {
		generator = new VetoableGenerator<Long>(new MillisGenerator(precision,
				low, high), tries) {
			@Override
			protected boolean tryValue(Long value) {
				return value != null;
			}
		};
	}

	@Override
	public Date next() {
		return new Date(generator.next());
	}

	private static Calendar calendar(long millis) {
		Calendar time = Calendar.getInstance();
		time.setTimeInMillis(millis);
		return time;
	}

	private static class MillisGenerator implements Generator<Long> {

		private final long precision;
		private final LongGenerator times;
		private final long low;
		private final long high;

		public MillisGenerator(TimeUnit precision, long low, long high) {
			this.precision = precision.toMillis(1);
			this.times = new LongGenerator(low, high);
			this.low = low;
			this.high = high;
		}

		@Override
		public Long next() {
			long millis = times.next() / precision * precision;
			long offset = daylightSavingOffset(millis) + timeZoneOffset(millis);
			long correctedMillis = millis - offset;
			return isOutOffBounds(correctedMillis)
					|| isOverflow(millis, correctedMillis) ? null
					: correctedMillis;
		}

		private boolean isOutOffBounds(long correctedMillis) {
			return correctedMillis < low || correctedMillis > high;
		}

		// TODO define this differently (signum changes near 0 without overflow)
		private boolean isOverflow(long millis, long correctedMillis) {
			return signum(correctedMillis) != signum(millis);
		}

		private int daylightSavingOffset(long millis) {
			return calendar(millis).get(Calendar.DST_OFFSET);
		}
		
		private int timeZoneOffset(long millis) {
			return calendar(millis).get(Calendar.ZONE_OFFSET);
		}
		
		@Override
		public String toString() {
			return format("%s[low=%s, high=%s, precision=%s", getClass().getSimpleName(), low, high, precision);
		}

	}
}