package net.lightapi.portal.instance.command.handler;

import com.networknt.config.JsonMapper;
import com.networknt.monad.Result;
import com.networknt.monad.Success;
import com.networknt.rpc.router.ServiceHandler;
import com.networknt.utility.Constants;
import com.networknt.utility.UuidUtil;
import io.cloudevents.CloudEvent;
import io.cloudevents.core.builder.CloudEventBuilder;
import io.undertow.server.HttpServerExchange;
import net.lightapi.portal.EventTypeUtil;
import net.lightapi.portal.PortalConstants;
import net.lightapi.portal.command.AbstractCommandHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Function;

@ServiceHandler(id = "lightapi.net/instance/compositeCreateInstanceApi/0.1.0")
public class CompositeCreateInstanceApi extends AbstractCommandHandler {
    private static final Logger logger = LoggerFactory.getLogger(CompositeCreateInstanceApi.class);

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected Result<Map<String, Object>> enrichInput(HttpServerExchange exchange, Map<String, Object> map) {
        String hostId = (String) map.get("hostId");
        String instanceId = (String) map.get("instanceId");
        String apiId = (String) map.get("apiId");
        String apiVersion = (String) map.get("apiVersion");
        String apiVersionId = getApiVersionId(hostId, apiId,  apiVersion);
        map.put("apiVersionId", apiVersionId);
        map.put("instanceApiId", getInstanceApiId(hostId, instanceId, apiVersionId));
        return Success.of(map);
    }

    @Override
    protected String getCloudEventType() {
        return PortalConstants.COMPOSITE_INSTANCE_API_CREATED_EVENT;
    }

    @Override
    protected String getCloudEventAggregateId(Map<String, Object> map) {
        return (String) map.get("instanceApiId");
    }

    @Override
    protected CloudEvent[] buildCloudEvent(Map<String, Object> map, String userId, String host, Number nonce) {
        List<CloudEvent> events = new ArrayList<>();
        events.add(buildApiCloudEvent(map, userId, host));
        events.add(buildApiVersionCloudEvent(map, userId, host));
        events.add(buildInstanceApiCloudEvent(map, userId, host));
        events.add(buildInstanceApiPathPrefixCloudEvent(map, userId, host));
        return events.toArray(new CloudEvent[0]);
    }

    private CloudEvent buildApiCloudEvent(Map<String, Object> map, String userId, String host) {
        String aggregateType = PortalConstants.AGGREGATE_API;
        String apiId = (String) map.get("apiId");
        String eventType = PortalConstants.API_CREATED_EVENT;

        if (apiId == null || apiId.isEmpty()) {
            throw new IllegalArgumentException("apiId is missing or empty in the provided data");
        }

        // first get the event store aggregate version with the aggregate id
        Map<String, Object> apiAggregateMap = new HashMap<>();
        String hostId = (String) map.get("hostId");
        apiAggregateMap.put("apiId", apiId);
        apiAggregateMap.put("hostId", hostId);
        String aggregateId = EventTypeUtil.getAggregateId(eventType, apiAggregateMap);
        int maxAggregateVersion = dbProvider.getMaxAggregateVersion(aggregateId);
        int newAggregateVersion = maxAggregateVersion + 1;

        CloudEventBuilder eventBuilder = CloudEventBuilder.v1()
                .withSource(PortalConstants.EVENT_SOURCE)
                .withType(eventType)
                .withId(UuidUtil.getUUID().toString())
                .withTime(OffsetDateTime.now())
                .withSubject(aggregateId)
                .withExtension(Constants.USER, userId)
                .withExtension(PortalConstants.NONCE, dbProvider.queryNonceByUserId(userId))
                .withExtension(Constants.HOST, host)
                .withExtension(PortalConstants.AGGREGATE_TYPE, aggregateType)
                .withExtension(PortalConstants.EVENT_AGGREGATE_VERSION, newAggregateVersion);

        String data = JsonMapper.toJson(prepareApiData(host, map));
        eventBuilder.withData("application/json", data.getBytes(StandardCharsets.UTF_8));

        if (logger.isTraceEnabled()) {
            logger.trace("Creating API event: user = {}, host = {}, type = {}, data = {}",
                    userId, host, eventType, data);
        }

        return eventBuilder.build();
    }

    private CloudEvent buildApiVersionCloudEvent(Map<String, Object> map, String userId, String host) {
        String aggregateType = PortalConstants.AGGREGATE_API_VERSION;
        String eventType = PortalConstants.API_VERSION_CREATED_EVENT;

        String apiVersionId = (String) map.get("apiVersionId");
        String apiId = (String) map.get("apiId");

        if (apiVersionId == null || apiVersionId.isEmpty()) {
            throw new IllegalArgumentException("apiVersionId is missing or empty in the provided data");
        }

        // first get the event store aggregate version with the aggregate id
        Map<String, Object> apiAggregateMap = new HashMap<>();
        apiAggregateMap.put("apiVersionId", apiVersionId);
        String aggregateId = EventTypeUtil.getAggregateId(eventType, apiAggregateMap);
        int maxAggregateVersion = dbProvider.getMaxAggregateVersion(aggregateId);
        int newAggregateVersion = maxAggregateVersion + 1;

        CloudEventBuilder eventBuilder = CloudEventBuilder.v1()
                .withSource(PortalConstants.EVENT_SOURCE)
                .withType(eventType)
                .withId(UuidUtil.getUUID().toString())
                .withTime(OffsetDateTime.now())
                .withSubject(aggregateId)
                .withExtension(Constants.USER, userId)
                .withExtension(PortalConstants.NONCE, dbProvider.queryNonceByUserId(userId))
                .withExtension(Constants.HOST, host)
                .withExtension(PortalConstants.AGGREGATE_TYPE, aggregateType)
                .withExtension(PortalConstants.EVENT_AGGREGATE_VERSION, newAggregateVersion);

        String data = JsonMapper.toJson(prepareApiVersionData(host, apiVersionId, apiId, map));
        eventBuilder.withData("application/json", data.getBytes(StandardCharsets.UTF_8));

        if (logger.isTraceEnabled()) {
            logger.trace("Creating API event: user = {}, host = {}, type = {}, data = {}",
                    userId, host, eventType, data);
        }

        return eventBuilder.build();
    }

    private CloudEvent buildInstanceApiCloudEvent(Map<String, Object> map, String userId, String host) {
        String aggregateType = PortalConstants.AGGREGATE_INSTANCE_API;
        String eventType = PortalConstants.INSTANCE_API_CREATED_EVENT;
        String instanceApiId = (String) map.get("instanceApiId");
        String apiVersionId = (String) map.get("apiVersionId");

        if (instanceApiId == null || instanceApiId.isEmpty()) {
            throw new IllegalArgumentException("instanceApiId is missing or empty in the provided data");
        }

        // first get the event store aggregate version with the aggregate id
        Map<String, Object> apiAggregateMap = new HashMap<>();
        apiAggregateMap.put("instanceApiId", instanceApiId);
        String aggregateId = EventTypeUtil.getAggregateId(eventType, apiAggregateMap);

        CloudEventBuilder eventBuilder = CloudEventBuilder.v1()
                .withSource(PortalConstants.EVENT_SOURCE)
                .withType(eventType)
                .withId(UuidUtil.getUUID().toString())
                .withTime(OffsetDateTime.now())
                .withSubject(aggregateId)
                .withExtension(Constants.USER, userId)
                .withExtension(PortalConstants.NONCE, dbProvider.queryNonceByUserId(userId))
                .withExtension(Constants.HOST, host)
                .withExtension(PortalConstants.AGGREGATE_TYPE, aggregateType)
                .withExtension(PortalConstants.EVENT_AGGREGATE_VERSION, (Number) map.get(PortalConstants.NEW_AGGREGATE_VERSION));

        String data = JsonMapper.toJson(prepareInstanceApiData(host, apiVersionId, instanceApiId, (String) map.get("instanceId")));
        eventBuilder.withData("application/json", data.getBytes(StandardCharsets.UTF_8));

        if (logger.isTraceEnabled()) {
            logger.trace("Creating Instance Api event: user = {}, host = {}, type = {}, data = {}",
                    userId, host, eventType, data);
        }

        return eventBuilder.build();
    }

    private CloudEvent buildInstanceApiPathPrefixCloudEvent(Map<String, Object> map, String userId, String host) {
        String aggregateType = PortalConstants.AGGREGATE_INSTANCE_API_PATH_PREFIX;
        String eventType = PortalConstants.INSTANCE_API_PATH_PREFIX_CREATED_EVENT;
        String instanceApiId = (String) map.get("instanceApiId");
        String pathPrefix = (String) map.get("pathPrefix");

        if (instanceApiId == null || instanceApiId.isEmpty()) {
            throw new IllegalArgumentException("instanceApiId is missing or empty in the provided data");
        }

        if (pathPrefix == null || pathPrefix.isEmpty()) {
            throw new IllegalArgumentException("pathPrefix is missing or empty in the provided data");
        }

        // first get the event store aggregate version with the aggregate id
        Map<String, Object> apiAggregateMap = new HashMap<>();
        apiAggregateMap.put("instanceApiId", instanceApiId);
        apiAggregateMap.put("pathPrefix", pathPrefix);
        String aggregateId = EventTypeUtil.getAggregateId(eventType, apiAggregateMap);
        int maxAggregateVersion = dbProvider.getMaxAggregateVersion(aggregateId);
        int newAggregateVersion = maxAggregateVersion + 1;

        CloudEventBuilder eventBuilder = CloudEventBuilder.v1()
                .withSource(PortalConstants.EVENT_SOURCE)
                .withType(eventType)
                .withId(UuidUtil.getUUID().toString())
                .withTime(OffsetDateTime.now())
                .withSubject(aggregateId)
                .withExtension(Constants.USER, userId)
                .withExtension(PortalConstants.NONCE, dbProvider.queryNonceByUserId(userId))
                .withExtension(Constants.HOST, host)
                .withExtension(PortalConstants.AGGREGATE_TYPE, aggregateType)
                .withExtension(PortalConstants.EVENT_AGGREGATE_VERSION, newAggregateVersion);

        String data = JsonMapper.toJson(prepareInstanceApiPathPrefixData(host, instanceApiId, pathPrefix));
        eventBuilder.withData("application/json", data.getBytes(StandardCharsets.UTF_8));

        if (logger.isTraceEnabled()) {
            logger.trace("Creating Instance Api event: user = {}, host = {}, type = {}, data = {}",
                    userId, host, eventType, data);
        }

        return eventBuilder.build();
    }

    private Map<String, Object> prepareApiData(String hostId, Map<String, Object> input) {
        Map<String, Object> apiData = new HashMap<>();
        apiData.put("hostId", hostId);
        applyIfNotNull("apiId", map -> map.get("apiId")).accept(input, apiData);
        applyIfNotNull("apiName", map -> map.get("apiName")).accept(input, apiData);
        applyIfNotNull("apiDesc", map -> map.get("apiDesc")).accept(input, apiData);
        applyIfNotNull("operationOwner", map -> map.get("operationOwner")).accept(input, apiData);
        applyIfNotNull("deliveryOwner", map -> map.get("deliveryOwner")).accept(input, apiData);
        applyIfNotNull("region", map -> map.get("region")).accept(input, apiData);
        applyIfNotNull("businessGroup", map -> map.get("businessGroup")).accept(input, apiData);
        applyIfNotNull("lob", map -> map.get("lob")).accept(input, apiData);
        applyIfNotNull("platform", map -> map.get("platform")).accept(input, apiData);
        applyIfNotNull("capability", map -> map.get("capability")).accept(input, apiData);
        applyIfNotNull("gitRepo", map -> map.get("gitRepo")).accept(input, apiData);
        applyIfNotNull("apiTags", map -> map.get("apiTags")).accept(input, apiData);
        applyIfNotNull("apiStatus", map -> map.get("apiStatus")).accept(input, apiData);
        return apiData;
    }

    private Map<String, Object> prepareApiVersionData(String hostId, String apiVersionId, String apiId, Map<String, Object> input) {
        Map<String, Object> apiVersionData = new HashMap<>();
        apiVersionData.put("hostId", hostId);
        apiVersionData.put("apiVersionId", apiVersionId);
        apiVersionData.put("apiId", apiId);
        applyIfNotNull("apiVersion", map -> map.get("apiVersion")).accept(input, apiVersionData);
        applyIfNotNull("apiType", map -> map.get("apiType")).accept(input, apiVersionData);
        applyIfNotNull("serviceId", map -> map.get("serviceId")).accept(input, apiVersionData);
        applyIfNotNull("apiVersionDesc", map -> map.get("apiVersionDesc")).accept(input, apiVersionData);
        return apiVersionData;
    }

    private Map<String, Object> prepareInstanceApiData(String hostId, String apiVersionId, String instanceApiId, String instanceId) {
        Map<String, Object> instanceApiData = new HashMap<>();
        instanceApiData.put("hostId", hostId);
        instanceApiData.put("apiVersionId", apiVersionId);
        instanceApiData.put("instanceApiId", instanceApiId);
        instanceApiData.put("instanceId", instanceId);
        return instanceApiData;
    }

    private Map<String, Object> prepareInstanceApiPathPrefixData(String hostId, String instanceApiId, String pathPrefix) {
        Map<String, Object> instanceApiPathPrefixData = new HashMap<>();
        instanceApiPathPrefixData.put("hostId", hostId);
        instanceApiPathPrefixData.put("instanceApiId", instanceApiId);
        instanceApiPathPrefixData.put("pathPrefix", pathPrefix);
        return instanceApiPathPrefixData;
    }

    private BiConsumer<Map<String, Object>, Map<String, Object>> applyIfNotNull(String key, Function<Map<String, Object>, Object> valueExtractor) {
        return (inputMap, outputMap) -> {
            Object value = valueExtractor.apply(inputMap);
            if (value != null) {
                outputMap.put(key, value);
            }
        };
    }

    private String getApiVersionId(String hostId, String apiId, String apiVersion) {
        String apiVersionId = dbProvider.queryApiVersionId(hostId, apiId, apiVersion);
        if (apiVersionId != null && !apiVersionId.isEmpty()) {
            return apiVersionId;
        }

        return UuidUtil.getUUID().toString();
    }

    private String getInstanceApiId(String hostId, String instanceId, String apiVersionId) {
        String existingInstanceApiId = dbProvider.getInstanceApiId(hostId, instanceId, apiVersionId);
        if (existingInstanceApiId != null && !existingInstanceApiId.isEmpty()) {
            return existingInstanceApiId;
        }
        return UuidUtil.getUUID().toString();
    }
}