package net.lightapi.portal.instance.command.handler;

import com.networknt.config.JsonMapper;
import com.networknt.db.provider.DbProvider;
import com.networknt.monad.Result;
import com.networknt.monad.Success;
import com.networknt.rpc.router.ServiceHandler;
import com.networknt.service.SingletonServiceFactory;
import com.networknt.utility.Constants;
import com.networknt.utility.UuidUtil;
import io.cloudevents.CloudEvent;
import io.cloudevents.core.builder.CloudEventBuilder;
import io.undertow.server.HttpServerExchange;
import net.lightapi.portal.EventTypeUtil;
import net.lightapi.portal.PortalConstants;
import net.lightapi.portal.command.AbstractCommandHandler;
import net.lightapi.portal.db.PortalDbProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@ServiceHandler(id = "lightapi.net/instance/compositeCreateInstanceApp/0.1.0")
public class CompositeCreateInstanceApp extends AbstractCommandHandler {
    private static final Logger logger = LoggerFactory.getLogger(CompositeCreateInstanceApp.class);
    public static PortalDbProvider dbProvider = (PortalDbProvider) SingletonServiceFactory.getBean(DbProvider.class);

    @Override
    protected String getCloudEventType() {
        return PortalConstants.COMPOSITE_INSTANCE_APP_CREATED_EVENT;
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    @Override
    protected String getCloudEventAggregateId(Map<String, Object> map) {
        return (String) map.get("instanceAppId");
    }

    @Override
    protected Result<Map<String, Object>> enrichInput(HttpServerExchange exchange, Map<String, Object> map) {
        String appId = (String) map.get("appId");
        String hostId = (String) map.get("hostId");
        String instanceId = (String) map.get("instanceId");
        String appVersion = (String) map.get("appVersion");
        // Generate New ID for instanceApp
        map.put("instanceAppId", getInstanceAppId(hostId, instanceId, appId, appVersion));
        return Success.of(map);
    }

    @Override
    protected CloudEvent[] buildCloudEvent(Map<String, Object> map, String userId, String host, Number nonce) {
        List<CloudEvent> events = new ArrayList<>();

        Map<String, Object> appData = new HashMap<>(map);
        appData.put("eventType", PortalConstants.APP_CREATED_EVENT);
        appData.put("appId", map.get("appId"));
        CloudEvent appEvent = buildCloudEventApp(appData, userId, host);
        events.add(appEvent);

        Map<String, Object> instanceAppData = new HashMap<>(map);
        instanceAppData.put("eventType", PortalConstants.INSTANCE_APP_CREATED_EVENT);
        CloudEvent instanceAppEvent = buildCloudEventInstanceApp(instanceAppData, userId, host);
        events.add(instanceAppEvent);

        return events.toArray(new CloudEvent[0]);
    }

    private CloudEvent buildCloudEventApp(Map<String, Object> map, String userId, String host) {
        String aggregateType = PortalConstants.AGGREGATE_APP;
        String appId = (String) map.get("appId");
        String eventType = (String) map.get("eventType");

        if (appId == null || appId.isEmpty()) {
            throw new IllegalArgumentException("appId is missing or empty in the provided data");
        }

        // first get the event store aggregate version with the aggregate id
        Map<String, Object> appAggregateMap = new HashMap<>();
        String hostId = (String) map.get("hostId");
        appAggregateMap.put("appId", appId);
        appAggregateMap.put("hostId", hostId);
        String aggregateId = EventTypeUtil.getAggregateId(eventType, appAggregateMap);
        int maxAggregateVersion = dbProvider.getMaxAggregateVersion(aggregateId);
        int newAggregateVersion = maxAggregateVersion + 1;

        CloudEventBuilder eventBuilder = CloudEventBuilder.v1()
                .withSource(PortalConstants.EVENT_SOURCE)
                .withType(eventType)
                .withId(UuidUtil.getUUID().toString())
                .withTime(OffsetDateTime.now())
                .withSubject(aggregateId)
                .withExtension(Constants.USER, userId)
                .withExtension(PortalConstants.NONCE, dbProvider.queryNonceByUserId(userId))
                .withExtension(Constants.HOST, host)
                .withExtension(PortalConstants.AGGREGATE_TYPE, aggregateType)
                .withExtension(PortalConstants.EVENT_AGGREGATE_VERSION, newAggregateVersion);

        String data = JsonMapper.toJson(map);
        eventBuilder.withData("application/json", data.getBytes(StandardCharsets.UTF_8));

        if (logger.isTraceEnabled()) {
            logger.trace("Creating App event: user = {}, host = {}, type = {}, data = {}",
                    userId, host, eventType, data);
        }

        return eventBuilder.build();
    }

    private CloudEvent buildCloudEventInstanceApp(Map<String, Object> map, String userId, String host) {
        String aggregateType = PortalConstants.AGGREGATE_INSTANCE_APP;
        String instanceAppId = (String) map.get("instanceAppId");
        String eventType = (String) map.get("eventType");

        if (instanceAppId == null || instanceAppId.isEmpty()) {
            throw new IllegalArgumentException("instanceAppId is missing or empty in the provided data");
        }

        Map<String, Object> instanceAppAggMap = new HashMap<>();
        instanceAppAggMap.put("instanceAppId", instanceAppId);
        String aggregateId = EventTypeUtil.getAggregateId(eventType, instanceAppAggMap);

        CloudEventBuilder eventBuilder = CloudEventBuilder.v1()
                .withSource(PortalConstants.EVENT_SOURCE)
                .withType(eventType)
                .withId(UuidUtil.getUUID().toString())
                .withTime(OffsetDateTime.now())
                .withSubject(aggregateId)
                .withExtension(Constants.USER, userId)
                .withExtension(PortalConstants.NONCE, dbProvider.queryNonceByUserId(userId))
                .withExtension(Constants.HOST, host)
                .withExtension(PortalConstants.AGGREGATE_TYPE, aggregateType)
                // Since populateAggregateVersion already sets the new aggregate version in the map, we just retrieve it here.
                .withExtension(PortalConstants.EVENT_AGGREGATE_VERSION, (Number) map.get(PortalConstants.NEW_AGGREGATE_VERSION));

        String data = JsonMapper.toJson(map);
        eventBuilder.withData("application/json", data.getBytes(StandardCharsets.UTF_8));

        if (logger.isTraceEnabled()) {
            logger.trace("Creating Instance App event: user = {}, host = {}, type = {}, data = {}",
                    userId, host, eventType, data);
        }

        return eventBuilder.build();
    }

    private static String getInstanceAppId(String hostId, String instanceId, String appId, String appVersion) {
        String instanceAppId = dbProvider.getInstanceAppId(hostId, instanceId, appId, appVersion);
        if (instanceAppId != null && !instanceAppId.isEmpty()) {
            return instanceAppId;
        }

        return UuidUtil.getUUID().toString();
    }
}