package net.lightapi.portal.instance.command.handler;

import com.networknt.config.JsonMapper;
import com.networknt.monad.Failure;
import com.networknt.monad.Result;
import com.networknt.monad.Success;
import com.networknt.rpc.router.ServiceHandler;
import com.networknt.status.HttpStatus;
import com.networknt.status.Status;
import io.undertow.server.HttpServerExchange;
import net.lightapi.portal.HybridQueryClient;
import net.lightapi.portal.PortalConstants;
import net.lightapi.portal.command.AbstractCommandHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Promotes data into an instance
 */
@ServiceHandler(id="lightapi.net/instance/promoteInstance/0.1.0")
public class PromoteInstance extends AbstractCommandHandler {
    private static final Logger logger = LoggerFactory.getLogger(PromoteInstance.class);

    @Override
    protected String getCloudEventType() {
        return PortalConstants.INSTANCE_PROMOTED_EVENT;
    }

    @Override
    protected String getCloudEventAggregateType() {
        return "Instance";
    }

    @Override
    protected String getCloudEventAggregateId(Map<String, Object> map) {
        return (String) map.get("instanceId");
    }

    @Override
    protected Result<Map<String, Object>> validateInput(
        HttpServerExchange exchange, Map<String, Object> map, String userId, String host) {
        final Set<String> applicableResourceTypes = Set.of();
        final Set<String> applicableConfigTypes = Set.of();
        final Set<String> applicablePropertyTypes = Set.of();
        Result<String> applicablePropertiesResult = HybridQueryClient.getApplicableConfigPropertiesForInstance(
            exchange, (String) map.get("hostId"), (String) map.get("instanceId"),
            applicableResourceTypes, applicableConfigTypes, applicablePropertyTypes,
            0, Integer.MAX_VALUE
        );

        if (applicablePropertiesResult.isFailure()) {
            return Failure.of(applicablePropertiesResult.getError());
        }


        return PromoteInstanceInputValidator.validate(map, applicablePropertiesResult.getResult());
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }

    static final class PromoteInstanceInputValidator {
        // Status Error Keys
        private static final String PROPERTIES_NOT_FOUND_ERROR_CODE = "ERR11210";
        private static final String PROPERTIES_NOT_FOUND_ERROR_MESSAGE = "APPLICABLE_PROPERTIES_NOT_FOUND";
        private static final String PROPERTIES_NOT_FOUND_ERROR_DESCRIPTION = "The system is not able to find any applicable properties. " +
            "Please ensure that the system is correctly configured and has applicable properties defined.";
        private static final String INVALID_PROMOTABLE_CONFIGS_ERROR_CODE = "ERR11310";
        private static final String INVALID_PROMOTABLE_CONFIGS_ERROR_DESCRIPTION =
            "The promotable instance that has been provided is not valid";

        // For DB data resolution
        private static final String INSTANCE_APPLICABLE_PROPERTIES_KEY = "instanceApplicableProperties";
        private static final String CONFIG_ID_KEY = "configId";
        private static final String CONFIG_NAME_KEY = "configName";
        private static final String PROPERTY_ID_KEY = "propertyId";
        private static final String PROPERTY_NAME_KEY = "propertyName";
        private static final String PROPERTY_VALUE_TYPE_KEY = "valueType";
        private static final String PROPERTY_RESOURCE_TYPE_KEY = "resourceType";

        // For Promotable Instance data resolution
        private static final String PROMOTABLE_RESOURCE_CONFIGS_KEY = "configs";
        private static final String PROMOTABLE_APIS_KEY = "apis";
        private static final String PROMOTABLE_APPS_KEY = "apps";
        private static final String RESOURCE_UID_KEY = "uid";
        private static final String RESOURCE_PATH_PREFIXES_KEY = "path_prefixes";

        // For Errors
        private static final String INSTANCE_CONFIG_ERRORS_RESULT_KEY = "instanceConfigErrors";
        private static final String API_ERROR_RESULT_KEY = "apiErrors";
        private static final String APP_ERRORS_RESULT_KEY = "appErrors";
        private static final String INSTANCE_CONFIG_PROPERTY_VALUE_ERROR_FORMAT =
            "Property value for Property Name %s is not valid for value type %s";
        private static final String INSTANCE_PROPERTY_NAME_NOT_FOUND_ERROR_FORMAT = "Property name %s not found";
        private static final String INVALID_API_UID_ERROR_FORMAT = "%s Invalid api uid";
        private static final String EMPTY_PATH_PREFIXES_ERROR_FORMAT = "%s empty api path prefixes provided";
        private static final String INVALID_APP_UID_ERROR_FORMAT = "%s Invalid app uid";
        private static final String EMPTY_INSTANCE_API_CONFIGS_ERROR_FORMAT = "%s api configs are empty";
        private static final String EMPTY_INSTANCE_APP_CONFIGS_ERROR_FORMAT = "%s app configs are empty";
        private static final String RESOURCE_PROPERTY_VALUE_ERROR_FORMAT =
            "%s Property value for Property Name %s is not valid for value type %s";
        private static final String INVALID_INSTANCE_API_RESOURCE_ERROR_FORMAT =
            "%s Property for Property Name %s is not valid for instance api";
        private static final String INVALID_PROPERTY_NAME_FOR_RESOURCE_ERROR_FORMAT = "%s Property name %s not found";
        private static final String INVALID_INSTANCE_APP_RESOURCE_ERROR_FORMAT =
            "%s Property for Property Name %s is not valid for instance app";
        private static final String INVALID_INSTANCE_APP_API_RESOURCE_ERROR_FORMAT =
            "%s Property for Property Name %s is not valid for instance app-api";
        private static final Set<String> INSTANCE_API_RESOURCE_TYPES = Set.of("api", "api|app_api", "all");
        private static final Set<String> INSTANCE_APP_RESOURCE_TYPES = Set.of("app", "app|app_api", "all");
        private static final Set<String> INSTANCE_APP_API_RESOURCE_TYPES = Set.of("app_api", "api|app_api", "app|app_api", "all");

        private PromoteInstanceInputValidator() {
            // prevent init
        }

        @SuppressWarnings("unchecked")
        public static Result<Map<String, Object>> validate(
            Map<String, Object> toValidate, String instanceApplicablePropertiesJson) {
            // All props from Data Source
            Map<String, Object> rawResultAsMap = JsonMapper.string2Map(instanceApplicablePropertiesJson);
            if (Objects.isNull(rawResultAsMap) || !rawResultAsMap.containsKey(INSTANCE_APPLICABLE_PROPERTIES_KEY)) {
                return Failure.of(propertiesNotFound());
            }

            List<Map<String, Object>> rawApplicableProperties =
                (List<Map<String, Object>>) rawResultAsMap.get(INSTANCE_APPLICABLE_PROPERTIES_KEY);

            List<InstanceProperty> sourceProperties = rawApplicableProperties.stream()
                .map(map -> new InstanceProperty(
                    (String) map.get(CONFIG_ID_KEY),
                    (String) map.get(PROPERTY_ID_KEY),
                    Objects.nonNull(map.get(CONFIG_NAME_KEY)) ? ((String) map.get(CONFIG_NAME_KEY)).trim() : "",
                    Objects.nonNull(map.get(PROPERTY_NAME_KEY)) ? ((String) map.get(PROPERTY_NAME_KEY)).trim() : "",
                    (String)  map.get(PROPERTY_VALUE_TYPE_KEY),
                    (String) map.get(PROPERTY_RESOURCE_TYPE_KEY)
                ))
                .toList();

            Map<String, InstanceProperty> sourcePropertiesByName = sourceProperties
                .stream()
                .filter(InstanceProperty::isValid)
                .collect(Collectors.toMap(
                    InstanceProperty::fullyQualifiedPropertyName,
                    Function.identity(),
                    (existing, replacement) -> existing
                ));

            Map<String, Object> toValidateInstanceConfigs =
                (Map<String, Object>) toValidate.get(PROMOTABLE_RESOURCE_CONFIGS_KEY);

            Map<String, Object> errorsMap = new HashMap<>();

            if (Objects.nonNull(toValidateInstanceConfigs) && !toValidateInstanceConfigs.isEmpty()) {
                var instanceConfigErrors = toValidateInstanceConfigs
                    .entrySet()
                    .stream()
                    .map(entry -> validateInstanceConfig(entry, sourcePropertiesByName))
                    .filter(errors -> !errors.isEmpty())
                    .flatMap(Collection::stream)
                    .toList();

                if (!instanceConfigErrors.isEmpty()) {
                    errorsMap.put(INSTANCE_CONFIG_ERRORS_RESULT_KEY, instanceConfigErrors);
                }
            }

            List<Map<String, Object>> toValidateApis = (List<Map<String, Object>>) toValidate.get(PROMOTABLE_APIS_KEY);
            if (Objects.nonNull(toValidateApis) && !toValidateApis.isEmpty()) {
                var apiErrors = toValidateApis.stream()
                    .map(api -> validateInstanceApi(api, sourcePropertiesByName))
                    .filter(e -> !e.isEmpty())
                    .flatMap(Collection::stream)
                    .toList();

                if (!apiErrors.isEmpty()) {
                    errorsMap.put(API_ERROR_RESULT_KEY, apiErrors);
                }
            }

            List<Map<String, Object>> toValidateApps = (List<Map<String, Object>>) toValidate.get(PROMOTABLE_APPS_KEY);
            if (Objects.nonNull(toValidateApps) && !toValidateApps.isEmpty()) {
                var appErrors = toValidateApps.stream()
                    .map(app -> validateInstanceApp(app, sourcePropertiesByName))
                    .filter(e -> !e.isEmpty())
                    .flatMap(Collection::stream)
                    .toList();

                if (!appErrors.isEmpty()) {
                    errorsMap.put(APP_ERRORS_RESULT_KEY, appErrors);
                }
            }

            return errorsMap.isEmpty()
                ? Success.of(toValidate)
                : Failure.of(invalidPromotableInstance(errorsMap));
        }

        private static Set<String> validateInstanceConfig(
            Map.Entry<String, Object> entry, Map<String, InstanceProperty> sourcePropertiesByName) {
            Set<String> errors = new HashSet<>();
            String key = entry.getKey();
            if (sourcePropertiesByName.containsKey(key)) {
                Object value = entry.getValue();
                String valueType = sourcePropertiesByName.get(key).valueType();
                boolean isValid = isValueValid(value, valueType);

                if (!isValid) {
                    errors.add(String.format(INSTANCE_CONFIG_PROPERTY_VALUE_ERROR_FORMAT, key, valueType));
                }

            } else {
                errors.add(String.format(INSTANCE_PROPERTY_NAME_NOT_FOUND_ERROR_FORMAT, key));
            }
            return errors;
        }

        @SuppressWarnings("unchecked")
        private static Set<String> validateInstanceApi(
            Map<String, Object> api, Map<String, InstanceProperty> sourcePropertiesByName) {
            Set<String> errors = new HashSet<>();
            String apiPrefix;
            var uid = (String) api.get(RESOURCE_UID_KEY);
            if (Objects.isNull(uid) || uid.isBlank() || uid.replace("-", "").isBlank()) {
                apiPrefix = "<<api(na)>>";
                errors.add(String.format(INVALID_API_UID_ERROR_FORMAT, apiPrefix));
            } else {
                apiPrefix = String.format("<<api(%s)>>", uid.trim());
            }

            var pathPrefixes = (List<String>) api.get(RESOURCE_PATH_PREFIXES_KEY);
            if (Objects.isNull(pathPrefixes) || pathPrefixes.isEmpty()) {
                errors.add(String.format(EMPTY_PATH_PREFIXES_ERROR_FORMAT, apiPrefix));
            }

            var configs = (Map<String, Object>) api.get(PROMOTABLE_RESOURCE_CONFIGS_KEY);
            errors.addAll(validateInstanceApiConfigs(configs, sourcePropertiesByName, apiPrefix));

            var apps = (List<Map<String, Object>>) api.get(PROMOTABLE_APPS_KEY);
            if (Objects.nonNull(apps) && !apps.isEmpty()) {
                var apiAppsErrors = apps.stream()
                    .map(app -> validateInstanceApiApp(app, sourcePropertiesByName, apiPrefix))
                    .filter(e -> !e.isEmpty())
                    .flatMap(Collection::stream)
                    .toList();

                errors.addAll(apiAppsErrors);
            }
            return errors;
        }

        @SuppressWarnings("unchecked")
        private static Set<String> validateInstanceApp(
            Map<String, Object> app, Map<String, InstanceProperty> sourcePropertiesByName) {
            Set<String> errors = new HashSet<>();
            String appPrefix;
            var uid = (String) app.get(RESOURCE_UID_KEY);
            if (Objects.isNull(uid) || uid.isBlank()) {
                appPrefix = "<<app(na)>>";
                errors.add(String.format(INVALID_APP_UID_ERROR_FORMAT, appPrefix));
            } else {
                appPrefix = String.format("<<app(%s)>>", uid.trim());
            }

            var configs = (Map<String, Object>) app.get(PROMOTABLE_RESOURCE_CONFIGS_KEY);
            errors.addAll(validateInstanceAppConfigs(configs, sourcePropertiesByName, appPrefix));
            return errors;
        }

        @SuppressWarnings("unchecked")
        private static Set<String> validateInstanceApiApp(Map<String, Object> app, Map<String, InstanceProperty> sourcePropertiesByName, String apiPrefix) {
            Set<String> errors = new HashSet<>();
            String apiAppPrefix;
            var uid = (String) app.get(RESOURCE_UID_KEY);
            if (Objects.isNull(uid) || uid.isBlank()) {
                apiAppPrefix = apiPrefix + "<<app(na)>>";
                errors.add(String.format(INVALID_APP_UID_ERROR_FORMAT, apiAppPrefix));
            } else {
                apiAppPrefix = String.format("%s<<app(%s)>>", apiPrefix, uid.trim());
            }

            var configs = (Map<String, Object>) app.get(PROMOTABLE_RESOURCE_CONFIGS_KEY);
            errors.addAll(validateInstanceAppApiConfigs(configs, sourcePropertiesByName, apiAppPrefix));
            return errors;
        }

        private static Set<String> validateInstanceApiConfigs(
            Map<String, Object> configs, Map<String, InstanceProperty> sourcePropertiesByName, String apiPrefix) {
            Set<String> errors = new HashSet<>();
            if (Objects.isNull(configs) || configs.isEmpty()) {
                errors.add(String.format(EMPTY_INSTANCE_API_CONFIGS_ERROR_FORMAT, apiPrefix));
            } else {
                var configErrors = configs.entrySet()
                    .stream()
                    .map(config -> validateInstanceApiConfig(config, sourcePropertiesByName, apiPrefix))
                    .filter(e -> !e.isEmpty())
                    .flatMap(Collection::stream)
                    .toList();

                errors.addAll(configErrors);
            }

            return errors;
        }

        private static Set<String> validateInstanceAppConfigs(
            Map<String, Object> configs, Map<String, InstanceProperty> sourcePropertiesByName, String appPrefix) {
            Set<String> errors = new HashSet<>();
            if (Objects.isNull(configs) || configs.isEmpty()) {
                errors.add(String.format(EMPTY_INSTANCE_APP_CONFIGS_ERROR_FORMAT, appPrefix));
            } else {
                var configErrors = configs.entrySet()
                    .stream()
                    .map(config -> validateInstanceAppConfig(config, sourcePropertiesByName, appPrefix))
                    .filter(e -> !e.isEmpty())
                    .flatMap(Collection::stream)
                    .toList();

                errors.addAll(configErrors);
            }

            return errors;
        }

        private static Set<String> validateInstanceAppApiConfigs(
            Map<String, Object> configs, Map<String, InstanceProperty> sourcePropertiesByName,
            String apiAppPrefix) {
            Set<String> errors = new HashSet<>();
            if (Objects.isNull(configs) || configs.isEmpty()) {
                errors.add(String.format(EMPTY_INSTANCE_APP_CONFIGS_ERROR_FORMAT, apiAppPrefix));
            } else {
                var configErrors = configs.entrySet()
                    .stream()
                    .map(config -> validateInstanceApiAppConfig(config, sourcePropertiesByName, apiAppPrefix))
                    .filter(e -> !e.isEmpty())
                    .flatMap(Collection::stream)
                    .toList();

                errors.addAll(configErrors);
            }

            return errors;
        }

        private static Set<String> validateInstanceApiConfig(
            Map.Entry<String, Object> entry,  Map<String, InstanceProperty> sourcePropertiesByName, String apiPrefix) {
            Set<String> errors = new HashSet<>();
            String key = entry.getKey();
            if (sourcePropertiesByName.containsKey(key)) {
                Object value = entry.getValue();
                String valueType = sourcePropertiesByName.get(key).valueType();
                boolean isValid = isValueValid(value, valueType);
                if (!isValid) {
                    errors.add(String.format(RESOURCE_PROPERTY_VALUE_ERROR_FORMAT, apiPrefix, key, valueType));
                }

                String resourceType = sourcePropertiesByName.get(key).resourceType();
                if (Objects.isNull(resourceType) || resourceType.isBlank() || !INSTANCE_API_RESOURCE_TYPES.contains(resourceType.toLowerCase())) {
                    errors.add(String.format(INVALID_INSTANCE_API_RESOURCE_ERROR_FORMAT, apiPrefix, key));
                }

            } else {
                errors.add(String.format(INVALID_PROPERTY_NAME_FOR_RESOURCE_ERROR_FORMAT, apiPrefix, key));
            }
            return errors;
        }

        private static Set<String> validateInstanceAppConfig(
            Map.Entry<String, Object> entry,  Map<String, InstanceProperty> sourcePropertiesByName, String appPrefix) {
            Set<String> errors = new HashSet<>();
            String key = entry.getKey();
            if (sourcePropertiesByName.containsKey(key)) {
                Object value = entry.getValue();
                String valueType = sourcePropertiesByName.get(key).valueType();
                boolean isValid = isValueValid(value, valueType);
                if (!isValid) {
                    errors.add(String.format(RESOURCE_PROPERTY_VALUE_ERROR_FORMAT, appPrefix, key, valueType));
                }

                String resourceType = sourcePropertiesByName.get(key).resourceType();
                if (Objects.isNull(resourceType) || resourceType.isBlank() || !INSTANCE_APP_RESOURCE_TYPES.contains(resourceType.toLowerCase())) {
                    errors.add(String.format(INVALID_INSTANCE_APP_RESOURCE_ERROR_FORMAT, appPrefix, key));
                }

            } else {
                errors.add(String.format(INVALID_PROPERTY_NAME_FOR_RESOURCE_ERROR_FORMAT, appPrefix, key));
            }
            return errors;
        }

        private static Set<String> validateInstanceApiAppConfig(
            Map.Entry<String, Object> entry, Map<String, InstanceProperty> sourcePropertiesByName, String apiAppPrefix) {
            Set<String> errors = new HashSet<>();
            String key = entry.getKey();
            if (sourcePropertiesByName.containsKey(key)) {
                Object value = entry.getValue();
                String valueType = sourcePropertiesByName.get(key).valueType();
                boolean isValid = isValueValid(value, valueType);
                if (!isValid) {
                    errors.add(String.format(RESOURCE_PROPERTY_VALUE_ERROR_FORMAT, apiAppPrefix, key, valueType));
                }

                String resourceType = sourcePropertiesByName.get(key).resourceType();
                if (Objects.isNull(resourceType) || resourceType.isBlank() || !INSTANCE_APP_API_RESOURCE_TYPES.contains(resourceType.toLowerCase())) {
                    errors.add(String.format(INVALID_INSTANCE_APP_API_RESOURCE_ERROR_FORMAT, apiAppPrefix, key));
                }

            } else {
                errors.add(String.format(INVALID_PROPERTY_NAME_FOR_RESOURCE_ERROR_FORMAT, apiAppPrefix, key));
            }
            return errors;
        }

        private static boolean isValueValid(Object value, String valueType) {
            if (Objects.isNull(value)) {
                return true;
            }

            return switch (valueType.toLowerCase()) {
                case "boolean" -> value instanceof Boolean;
                case "integer" -> value instanceof Short || value instanceof Integer || value instanceof Long;
                case "float" -> value instanceof Short || value instanceof Integer || value instanceof Long
                    || value instanceof Float || value instanceof Double;
                case "map" -> value instanceof Map<?,?>;
                case "list" -> value instanceof Collection<?>;
                case "string" -> value instanceof CharSequence;
                default -> false;
            };
        }

        private static Status propertiesNotFound() {
            return new Status(
                HttpStatus.BAD_REQUEST.value(),
                PROPERTIES_NOT_FOUND_ERROR_CODE,
                PROPERTIES_NOT_FOUND_ERROR_MESSAGE,
                PROPERTIES_NOT_FOUND_ERROR_DESCRIPTION
            );
        }

        private static Status invalidPromotableInstance(Map<String, Object> errorsMap) {
            return new Status(
                HttpStatus.BAD_REQUEST.value(),
                INVALID_PROMOTABLE_CONFIGS_ERROR_CODE,
                JsonMapper.toJson(errorsMap),
                INVALID_PROMOTABLE_CONFIGS_ERROR_DESCRIPTION
            );
        }

        private record InstanceProperty(
            String configId, String propertyId, String configName, String propertyName, String valueType, String resourceType) {

            @Override
            public boolean equals(Object o) {
                if (this == o) return true;
                if (o == null || getClass() != o.getClass()) return false;
                InstanceProperty that = (InstanceProperty) o;
                return Objects.equals(configId, that.configId) && Objects.equals(propertyId, that.propertyId);
            }

            @Override
            public int hashCode() {
                return Objects.hash(configId, propertyId);
            }

            public boolean isValid() {
                return Objects.nonNull(configId) && !configId.isBlank()
                    && Objects.nonNull(propertyId) && !propertyId.isBlank()
                    && Objects.nonNull(configName) && !configName.isBlank()
                    && Objects.nonNull(propertyName) && !propertyName.isBlank()
                    && Objects.nonNull(valueType) && !valueType.isBlank();
            }

            public String fullyQualifiedPropertyName() {
                if (Objects.nonNull(configName) && !configName.isBlank()
                    && Objects.nonNull(propertyName) && !propertyName.isBlank()) {
                    return configName.concat(".").concat(propertyName);
                }

                return "";
            }
        }
    }
}
