/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.bedrock;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.Jwts;
import net.lenni0451.commons.httpclient.HttpClient;
import net.lenni0451.commons.httpclient.constants.Headers;
import net.lenni0451.commons.httpclient.requests.impl.PostRequest;
import net.raphimc.minecraftauth.MinecraftAuth;
import net.raphimc.minecraftauth.responsehandler.MinecraftResponseHandler;
import net.raphimc.minecraftauth.step.AbstractStep;
import net.raphimc.minecraftauth.step.xbl.StepXblXstsToken;
import net.raphimc.minecraftauth.util.CryptUtil;
import net.raphimc.minecraftauth.util.JsonContent;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECGenParameterSpec;
import java.util.Base64;
import java.util.Map;
import java.util.UUID;

public class StepMCChain extends AbstractStep<StepXblXstsToken.XblXsts<?>, StepMCChain.MCChain> {
    public static final String MINECRAFT_LOGIN_URL = "https://multiplayer.minecraft.net/authentication";
    private static final String MOJANG_PUBLIC_KEY_BASE64 = "MHYwEAYHKoZIzj0CAQYFK4EEACIDYgAECRXueJeTDqNRRgJi/vlRufByu/2G0i2Ebt6YMar5QX/R0DIIyrJMcUpruK4QveTfJSTp3Shlq4Gk34cD/4GUWwkv0DVuzeuB+tXija7HBxii03NHDbPAD0AKnLr2wdAp";
    public static final ECPublicKey MOJANG_PUBLIC_KEY = CryptUtil.publicKeyEcFromBase64(MOJANG_PUBLIC_KEY_BASE64);
    private static final int CLOCK_SKEW = 60;

    public StepMCChain(final AbstractStep<?, ? extends StepXblXstsToken.XblXsts<?>> prevStep) {
        super("mcChain", (AbstractStep<?, StepXblXstsToken.XblXsts<?>>) prevStep);
    }

    @Override
    public MCChain applyStep(final HttpClient httpClient, final StepXblXstsToken.XblXsts<?> xblXsts) throws Exception {
        MinecraftAuth.LOGGER.info("Authenticating with Minecraft Services...");
        final KeyPairGenerator secp384r1 = KeyPairGenerator.getInstance("EC");
        secp384r1.initialize(new ECGenParameterSpec("secp384r1"));
        final KeyPair ecdsa384KeyPair = secp384r1.generateKeyPair();
        final ECPublicKey publicKey = (ECPublicKey) ecdsa384KeyPair.getPublic();
        final ECPrivateKey privateKey = (ECPrivateKey) ecdsa384KeyPair.getPrivate();
        final JsonObject postData = new JsonObject();
        postData.addProperty("identityPublicKey", Base64.getEncoder().encodeToString(publicKey.getEncoded()));
        final PostRequest postRequest = new PostRequest(MINECRAFT_LOGIN_URL);
        postRequest.setContent(new JsonContent(postData));
        postRequest.setHeader(Headers.AUTHORIZATION, "XBL3.0 x=" + xblXsts.getServiceToken());
        final JsonObject obj = httpClient.execute(postRequest, new MinecraftResponseHandler());
        final JsonArray chain = obj.getAsJsonArray("chain");
        if (chain.size() != 2) {
            throw new IllegalStateException("Invalid chain size");
        }
        final Jws<Claims> mojangJwt = Jwts.parser().clockSkewSeconds(CLOCK_SKEW).verifyWith(MOJANG_PUBLIC_KEY).build().parseSignedClaims(chain.get(0).getAsString());
        final ECPublicKey mojangJwtPublicKey = CryptUtil.publicKeyEcFromBase64(mojangJwt.getPayload().get("identityPublicKey", String.class));
        final Jws<Claims> identityJwt = Jwts.parser().clockSkewSeconds(CLOCK_SKEW).verifyWith(mojangJwtPublicKey).build().parseSignedClaims(chain.get(1).getAsString());
        final Map<String, Object> extraData = identityJwt.getPayload().get("extraData", Map.class);
        final String xuid = (String) extraData.get("XUID");
        final UUID id = UUID.fromString((String) extraData.get("identity"));
        final String displayName = (String) extraData.get("displayName");
        if (!extraData.containsKey("titleId")) {
            MinecraftAuth.LOGGER.warn("Minecraft chain does not contain titleId! You might get kicked from some servers");
        }
        final MCChain mcChain = new MCChain(publicKey, privateKey, chain.get(0).getAsString(), chain.get(1).getAsString(), xuid, id, displayName, xblXsts);
        MinecraftAuth.LOGGER.info("Got MC Chain, name: " + mcChain.displayName + ", uuid: " + mcChain.id + ", xuid: " + mcChain.xuid);
        return mcChain;
    }

    @Override
    public MCChain fromJson(final JsonObject json) {
        final StepXblXstsToken.XblXsts<?> xblXsts = this.prevStep != null ? this.prevStep.fromJson(json.getAsJsonObject(this.prevStep.name)) : null;
        return new MCChain(CryptUtil.publicKeyEcFromBase64(json.get("publicKey").getAsString()), CryptUtil.privateKeyEcFromBase64(json.get("privateKey").getAsString()), json.get("mojangJwt").getAsString(), json.get("identityJwt").getAsString(), json.get("xuid").getAsString(), UUID.fromString(json.get("id").getAsString()), json.get("displayName").getAsString(), xblXsts);
    }

    @Override
    public JsonObject toJson(final MCChain mcChain) {
        final JsonObject json = new JsonObject();
        json.addProperty("publicKey", Base64.getEncoder().encodeToString(mcChain.publicKey.getEncoded()));
        json.addProperty("privateKey", Base64.getEncoder().encodeToString(mcChain.privateKey.getEncoded()));
        json.addProperty("mojangJwt", mcChain.mojangJwt);
        json.addProperty("identityJwt", mcChain.identityJwt);
        json.addProperty("xuid", mcChain.xuid);
        json.addProperty("id", mcChain.id.toString());
        json.addProperty("displayName", mcChain.displayName);
        if (this.prevStep != null) json.add(this.prevStep.name, this.prevStep.toJson(mcChain.xblXsts));
        return json;
    }


    public static final class MCChain extends AbstractStep.StepResult<StepXblXstsToken.XblXsts<?>> {
        private final ECPublicKey publicKey;
        private final ECPrivateKey privateKey;
        private final String mojangJwt;
        private final String identityJwt;
        private final String xuid;
        private final UUID id;
        private final String displayName;
        private final StepXblXstsToken.XblXsts<?> xblXsts;

        @Override
        protected StepXblXstsToken.XblXsts<?> prevResult() {
            return this.xblXsts;
        }

        public MCChain(final ECPublicKey publicKey, final ECPrivateKey privateKey, final String mojangJwt, final String identityJwt, final String xuid, final UUID id, final String displayName, final StepXblXstsToken.XblXsts<?> xblXsts) {
            this.publicKey = publicKey;
            this.privateKey = privateKey;
            this.mojangJwt = mojangJwt;
            this.identityJwt = identityJwt;
            this.xuid = xuid;
            this.id = id;
            this.displayName = displayName;
            this.xblXsts = xblXsts;
        }

        public ECPublicKey getPublicKey() {
            return this.publicKey;
        }

        public ECPrivateKey getPrivateKey() {
            return this.privateKey;
        }

        public String getMojangJwt() {
            return this.mojangJwt;
        }

        public String getIdentityJwt() {
            return this.identityJwt;
        }

        public String getXuid() {
            return this.xuid;
        }

        public UUID getId() {
            return this.id;
        }

        public String getDisplayName() {
            return this.displayName;
        }

        public StepXblXstsToken.XblXsts<?> getXblXsts() {
            return this.xblXsts;
        }

        @Override
        public String toString() {
            return "StepMCChain.MCChain(publicKey=" + this.getPublicKey() + ", privateKey=" + this.getPrivateKey() + ", mojangJwt=" + this.getMojangJwt() + ", identityJwt=" + this.getIdentityJwt() + ", xuid=" + this.getXuid() + ", id=" + this.getId() + ", displayName=" + this.getDisplayName() + ", xblXsts=" + this.getXblXsts() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepMCChain.MCChain)) return false;
            final StepMCChain.MCChain other = (StepMCChain.MCChain) o;
            if (!other.canEqual((Object) this)) return false;
            final Object this$publicKey = this.getPublicKey();
            final Object other$publicKey = other.getPublicKey();
            if (this$publicKey == null ? other$publicKey != null : !this$publicKey.equals(other$publicKey)) return false;
            final Object this$privateKey = this.getPrivateKey();
            final Object other$privateKey = other.getPrivateKey();
            if (this$privateKey == null ? other$privateKey != null : !this$privateKey.equals(other$privateKey)) return false;
            final Object this$mojangJwt = this.getMojangJwt();
            final Object other$mojangJwt = other.getMojangJwt();
            if (this$mojangJwt == null ? other$mojangJwt != null : !this$mojangJwt.equals(other$mojangJwt)) return false;
            final Object this$identityJwt = this.getIdentityJwt();
            final Object other$identityJwt = other.getIdentityJwt();
            if (this$identityJwt == null ? other$identityJwt != null : !this$identityJwt.equals(other$identityJwt)) return false;
            final Object this$xuid = this.getXuid();
            final Object other$xuid = other.getXuid();
            if (this$xuid == null ? other$xuid != null : !this$xuid.equals(other$xuid)) return false;
            final Object this$id = this.getId();
            final Object other$id = other.getId();
            if (this$id == null ? other$id != null : !this$id.equals(other$id)) return false;
            final Object this$displayName = this.getDisplayName();
            final Object other$displayName = other.getDisplayName();
            if (this$displayName == null ? other$displayName != null : !this$displayName.equals(other$displayName)) return false;
            final Object this$xblXsts = this.getXblXsts();
            final Object other$xblXsts = other.getXblXsts();
            if (this$xblXsts == null ? other$xblXsts != null : !this$xblXsts.equals(other$xblXsts)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepMCChain.MCChain;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final Object $publicKey = this.getPublicKey();
            result = result * PRIME + ($publicKey == null ? 43 : $publicKey.hashCode());
            final Object $privateKey = this.getPrivateKey();
            result = result * PRIME + ($privateKey == null ? 43 : $privateKey.hashCode());
            final Object $mojangJwt = this.getMojangJwt();
            result = result * PRIME + ($mojangJwt == null ? 43 : $mojangJwt.hashCode());
            final Object $identityJwt = this.getIdentityJwt();
            result = result * PRIME + ($identityJwt == null ? 43 : $identityJwt.hashCode());
            final Object $xuid = this.getXuid();
            result = result * PRIME + ($xuid == null ? 43 : $xuid.hashCode());
            final Object $id = this.getId();
            result = result * PRIME + ($id == null ? 43 : $id.hashCode());
            final Object $displayName = this.getDisplayName();
            result = result * PRIME + ($displayName == null ? 43 : $displayName.hashCode());
            final Object $xblXsts = this.getXblXsts();
            result = result * PRIME + ($xblXsts == null ? 43 : $xblXsts.hashCode());
            return result;
        }
    }
}
