/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.edu;

import com.google.gson.JsonObject;
import net.lenni0451.commons.httpclient.HttpClient;
import net.lenni0451.commons.httpclient.requests.impl.PostRequest;
import net.raphimc.minecraftauth.MinecraftAuth;
import net.raphimc.minecraftauth.responsehandler.MinecraftEduServicesResponseHandler;
import net.raphimc.minecraftauth.step.AbstractStep;
import net.raphimc.minecraftauth.step.msa.StepMsaToken;
import net.raphimc.minecraftauth.util.JsonContent;

public class StepEduJWT extends AbstractStep<StepMsaToken.MsaToken, StepEduJWT.EduJWT> {
    public static final String MINECRAFT_LOGIN_URL = "https://login.minecrafteduservices.com/v2/signin";
    private final String version;
    private final int buildNumber;
    private final int protocolVersion;
    private final String platform;

    public StepEduJWT(final AbstractStep<?, StepMsaToken.MsaToken> prevStep, final String version, final int protocolVersion, final String platform) {
        super("eduJwt", prevStep);
        final String[] versionParts = version.split("\\.");
        if (versionParts.length != 3) {
            throw new IllegalArgumentException("Invalid version: " + version);
        }
        this.version = version;
        this.buildNumber = Integer.parseInt(versionParts[0]) * 10000000 + Integer.parseInt(versionParts[1]) * 100000 + Integer.parseInt(versionParts[2]) * 1000;
        this.protocolVersion = protocolVersion;
        this.platform = platform;
    }

    @Override
    public EduJWT applyStep(final HttpClient httpClient, final StepMsaToken.MsaToken msaToken) throws Exception {
        MinecraftAuth.LOGGER.info("Authenticating with Minecraft Education Services...");
        final JsonObject postData = new JsonObject();
        postData.addProperty("accessToken", msaToken.getAccessToken());
        postData.addProperty("build", this.buildNumber);
        postData.addProperty("clientVersion", this.protocolVersion);
        postData.addProperty("displayVersion", this.version);
        postData.addProperty("identityToken", msaToken.getAccessToken());
        postData.addProperty("platform", this.platform);
        final PostRequest postRequest = new PostRequest(MINECRAFT_LOGIN_URL);
        postRequest.setContent(new JsonContent(postData));
        final JsonObject obj = httpClient.execute(postRequest, new MinecraftEduServicesResponseHandler());
        final EduJWT eduJwt = new EduJWT(obj.get("response").getAsString(), msaToken);
        MinecraftAuth.LOGGER.info("Got Edu JWT");
        return eduJwt;
    }

    @Override
    public EduJWT fromJson(final JsonObject json) {
        final StepMsaToken.MsaToken msaToken = this.prevStep != null ? this.prevStep.fromJson(json.getAsJsonObject(this.prevStep.name)) : null;
        return new EduJWT(json.get("jwt").getAsString(), msaToken);
    }

    @Override
    public JsonObject toJson(final EduJWT eduJWT) {
        final JsonObject json = new JsonObject();
        json.addProperty("jwt", eduJWT.jwt);
        if (this.prevStep != null) json.add(this.prevStep.name, this.prevStep.toJson(eduJWT.msaToken));
        return json;
    }


    public static final class EduJWT extends AbstractStep.StepResult<StepMsaToken.MsaToken> {
        private final String jwt;
        private final StepMsaToken.MsaToken msaToken;

        @Override
        protected StepMsaToken.MsaToken prevResult() {
            return this.msaToken;
        }

        public EduJWT(final String jwt, final StepMsaToken.MsaToken msaToken) {
            this.jwt = jwt;
            this.msaToken = msaToken;
        }

        public String getJwt() {
            return this.jwt;
        }

        public StepMsaToken.MsaToken getMsaToken() {
            return this.msaToken;
        }

        @Override
        public String toString() {
            return "StepEduJWT.EduJWT(jwt=" + this.getJwt() + ", msaToken=" + this.getMsaToken() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepEduJWT.EduJWT)) return false;
            final StepEduJWT.EduJWT other = (StepEduJWT.EduJWT) o;
            if (!other.canEqual((Object) this)) return false;
            final Object this$jwt = this.getJwt();
            final Object other$jwt = other.getJwt();
            if (this$jwt == null ? other$jwt != null : !this$jwt.equals(other$jwt)) return false;
            final Object this$msaToken = this.getMsaToken();
            final Object other$msaToken = other.getMsaToken();
            if (this$msaToken == null ? other$msaToken != null : !this$msaToken.equals(other$msaToken)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepEduJWT.EduJWT;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final Object $jwt = this.getJwt();
            result = result * PRIME + ($jwt == null ? 43 : $jwt.hashCode());
            final Object $msaToken = this.getMsaToken();
            result = result * PRIME + ($msaToken == null ? 43 : $msaToken.hashCode());
            return result;
        }
    }
}
