/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.java;

import com.google.gson.JsonObject;
import net.lenni0451.commons.httpclient.HttpClient;
import net.lenni0451.commons.httpclient.constants.Headers;
import net.lenni0451.commons.httpclient.requests.impl.GetRequest;
import net.raphimc.minecraftauth.MinecraftAuth;
import net.raphimc.minecraftauth.responsehandler.MinecraftResponseHandler;
import net.raphimc.minecraftauth.step.AbstractStep;
import net.raphimc.minecraftauth.util.UuidUtil;
import java.util.UUID;

public class StepMCProfile extends AbstractStep<StepMCToken.MCToken, StepMCProfile.MCProfile> {
    public static final String MINECRAFT_PROFILE_URL = "https://api.minecraftservices.com/minecraft/profile";

    public StepMCProfile(final AbstractStep<?, StepMCToken.MCToken> prevStep) {
        super("mcProfile", prevStep);
    }

    @Override
    public MCProfile applyStep(final HttpClient httpClient, final StepMCToken.MCToken mcToken) throws Exception {
        MinecraftAuth.LOGGER.info("Getting profile...");
        final GetRequest getRequest = new GetRequest(MINECRAFT_PROFILE_URL);
        getRequest.setHeader(Headers.AUTHORIZATION, mcToken.getTokenType() + " " + mcToken.getAccessToken());
        final JsonObject obj = httpClient.execute(getRequest, new MinecraftResponseHandler());
        final MCProfile mcProfile = new MCProfile(UuidUtil.fromLenientString(obj.get("id").getAsString()), obj.get("name").getAsString(), obj.get("skins").getAsJsonArray().get(0).getAsJsonObject().get("url").getAsString(), mcToken);
        MinecraftAuth.LOGGER.info("Got MC Profile, name: " + mcProfile.name + ", uuid: " + mcProfile.id);
        return mcProfile;
    }

    @Override
    public MCProfile fromJson(final JsonObject json) {
        final StepMCToken.MCToken mcToken = this.prevStep != null ? this.prevStep.fromJson(json.getAsJsonObject(this.prevStep.name)) : null;
        return new MCProfile(UUID.fromString(json.get("id").getAsString()), json.get("name").getAsString(), json.get("skinUrl").getAsString(), mcToken);
    }

    @Override
    public JsonObject toJson(final MCProfile mcProfile) {
        final JsonObject json = new JsonObject();
        json.addProperty("id", mcProfile.id.toString());
        json.addProperty("name", mcProfile.name);
        json.addProperty("skinUrl", mcProfile.skinUrl);
        if (this.prevStep != null) json.add(this.prevStep.name, this.prevStep.toJson(mcProfile.mcToken));
        return json;
    }


    public static final class MCProfile extends AbstractStep.StepResult<StepMCToken.MCToken> {
        private final UUID id;
        private final String name;
        private final String skinUrl;
        private final StepMCToken.MCToken mcToken;

        @Override
        protected StepMCToken.MCToken prevResult() {
            return this.mcToken;
        }

        public MCProfile(final UUID id, final String name, final String skinUrl, final StepMCToken.MCToken mcToken) {
            this.id = id;
            this.name = name;
            this.skinUrl = skinUrl;
            this.mcToken = mcToken;
        }

        public UUID getId() {
            return this.id;
        }

        public String getName() {
            return this.name;
        }

        public String getSkinUrl() {
            return this.skinUrl;
        }

        public StepMCToken.MCToken getMcToken() {
            return this.mcToken;
        }

        @Override
        public String toString() {
            return "StepMCProfile.MCProfile(id=" + this.getId() + ", name=" + this.getName() + ", skinUrl=" + this.getSkinUrl() + ", mcToken=" + this.getMcToken() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepMCProfile.MCProfile)) return false;
            final StepMCProfile.MCProfile other = (StepMCProfile.MCProfile) o;
            if (!other.canEqual((Object) this)) return false;
            final Object this$id = this.getId();
            final Object other$id = other.getId();
            if (this$id == null ? other$id != null : !this$id.equals(other$id)) return false;
            final Object this$name = this.getName();
            final Object other$name = other.getName();
            if (this$name == null ? other$name != null : !this$name.equals(other$name)) return false;
            final Object this$skinUrl = this.getSkinUrl();
            final Object other$skinUrl = other.getSkinUrl();
            if (this$skinUrl == null ? other$skinUrl != null : !this$skinUrl.equals(other$skinUrl)) return false;
            final Object this$mcToken = this.getMcToken();
            final Object other$mcToken = other.getMcToken();
            if (this$mcToken == null ? other$mcToken != null : !this$mcToken.equals(other$mcToken)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepMCProfile.MCProfile;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final Object $id = this.getId();
            result = result * PRIME + ($id == null ? 43 : $id.hashCode());
            final Object $name = this.getName();
            result = result * PRIME + ($name == null ? 43 : $name.hashCode());
            final Object $skinUrl = this.getSkinUrl();
            result = result * PRIME + ($skinUrl == null ? 43 : $skinUrl.hashCode());
            final Object $mcToken = this.getMcToken();
            result = result * PRIME + ($mcToken == null ? 43 : $mcToken.hashCode());
            return result;
        }
    }
}
