/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.java;

import com.google.gson.JsonObject;
import net.lenni0451.commons.httpclient.HttpClient;
import net.lenni0451.commons.httpclient.requests.impl.PostRequest;
import net.raphimc.minecraftauth.MinecraftAuth;
import net.raphimc.minecraftauth.responsehandler.MinecraftResponseHandler;
import net.raphimc.minecraftauth.step.AbstractStep;
import net.raphimc.minecraftauth.step.xbl.StepXblXstsToken;
import net.raphimc.minecraftauth.util.JsonContent;
import java.time.Instant;
import java.time.ZoneId;

public class StepMCToken extends AbstractStep<StepXblXstsToken.XblXsts<?>, StepMCToken.MCToken> {
    public static final String MINECRAFT_LOGIN_URL = "https://api.minecraftservices.com/launcher/login";

    public StepMCToken(final AbstractStep<?, ? extends StepXblXstsToken.XblXsts<?>> prevStep) {
        super("mcToken", (AbstractStep<?, StepXblXstsToken.XblXsts<?>>) prevStep);
    }

    @Override
    public MCToken applyStep(final HttpClient httpClient, final StepXblXstsToken.XblXsts<?> xblXsts) throws Exception {
        MinecraftAuth.LOGGER.info("Authenticating with Minecraft Services...");
        final JsonObject postData = new JsonObject();
        postData.addProperty("platform", "PC_LAUNCHER");
        postData.addProperty("xtoken", "XBL3.0 x=" + xblXsts.getServiceToken());
        final PostRequest postRequest = new PostRequest(MINECRAFT_LOGIN_URL);
        postRequest.setContent(new JsonContent(postData));
        final JsonObject obj = httpClient.execute(postRequest, new MinecraftResponseHandler());
        final MCToken mcToken = new MCToken(obj.get("access_token").getAsString(), obj.get("token_type").getAsString(), System.currentTimeMillis() + obj.get("expires_in").getAsLong() * 1000, xblXsts);
        MinecraftAuth.LOGGER.info("Got MC Token, expires: " + Instant.ofEpochMilli(mcToken.getExpireTimeMs()).atZone(ZoneId.systemDefault()));
        return mcToken;
    }

    @Override
    public MCToken fromJson(final JsonObject json) {
        final StepXblXstsToken.XblXsts<?> xblXsts = this.prevStep != null ? this.prevStep.fromJson(json.getAsJsonObject(this.prevStep.name)) : null;
        return new MCToken(json.get("accessToken").getAsString(), json.get("tokenType").getAsString(), json.get("expireTimeMs").getAsLong(), xblXsts);
    }

    @Override
    public JsonObject toJson(final MCToken mcToken) {
        final JsonObject json = new JsonObject();
        json.addProperty("accessToken", mcToken.accessToken);
        json.addProperty("tokenType", mcToken.tokenType);
        json.addProperty("expireTimeMs", mcToken.expireTimeMs);
        if (this.prevStep != null) json.add(this.prevStep.name, this.prevStep.toJson(mcToken.xblXsts));
        return json;
    }


    public static final class MCToken extends AbstractStep.StepResult<StepXblXstsToken.XblXsts<?>> {
        private final String accessToken;
        private final String tokenType;
        private final long expireTimeMs;
        private final StepXblXstsToken.XblXsts<?> xblXsts;

        @Override
        protected StepXblXstsToken.XblXsts<?> prevResult() {
            return this.xblXsts;
        }

        @Override
        public boolean isExpired() {
            return this.expireTimeMs <= System.currentTimeMillis();
        }

        public MCToken(final String accessToken, final String tokenType, final long expireTimeMs, final StepXblXstsToken.XblXsts<?> xblXsts) {
            this.accessToken = accessToken;
            this.tokenType = tokenType;
            this.expireTimeMs = expireTimeMs;
            this.xblXsts = xblXsts;
        }

        public String getAccessToken() {
            return this.accessToken;
        }

        public String getTokenType() {
            return this.tokenType;
        }

        public long getExpireTimeMs() {
            return this.expireTimeMs;
        }

        public StepXblXstsToken.XblXsts<?> getXblXsts() {
            return this.xblXsts;
        }

        @Override
        public String toString() {
            return "StepMCToken.MCToken(accessToken=" + this.getAccessToken() + ", tokenType=" + this.getTokenType() + ", expireTimeMs=" + this.getExpireTimeMs() + ", xblXsts=" + this.getXblXsts() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepMCToken.MCToken)) return false;
            final StepMCToken.MCToken other = (StepMCToken.MCToken) o;
            if (!other.canEqual((Object) this)) return false;
            if (this.getExpireTimeMs() != other.getExpireTimeMs()) return false;
            final Object this$accessToken = this.getAccessToken();
            final Object other$accessToken = other.getAccessToken();
            if (this$accessToken == null ? other$accessToken != null : !this$accessToken.equals(other$accessToken)) return false;
            final Object this$tokenType = this.getTokenType();
            final Object other$tokenType = other.getTokenType();
            if (this$tokenType == null ? other$tokenType != null : !this$tokenType.equals(other$tokenType)) return false;
            final Object this$xblXsts = this.getXblXsts();
            final Object other$xblXsts = other.getXblXsts();
            if (this$xblXsts == null ? other$xblXsts != null : !this$xblXsts.equals(other$xblXsts)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepMCToken.MCToken;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final long $expireTimeMs = this.getExpireTimeMs();
            result = result * PRIME + (int) ($expireTimeMs >>> 32 ^ $expireTimeMs);
            final Object $accessToken = this.getAccessToken();
            result = result * PRIME + ($accessToken == null ? 43 : $accessToken.hashCode());
            final Object $tokenType = this.getTokenType();
            result = result * PRIME + ($tokenType == null ? 43 : $tokenType.hashCode());
            final Object $xblXsts = this.getXblXsts();
            result = result * PRIME + ($xblXsts == null ? 43 : $xblXsts.hashCode());
            return result;
        }
    }
}
