/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.java;

import com.google.gson.JsonObject;
import net.lenni0451.commons.httpclient.HttpClient;
import net.lenni0451.commons.httpclient.constants.ContentTypes;
import net.lenni0451.commons.httpclient.constants.Headers;
import net.lenni0451.commons.httpclient.content.impl.StringContent;
import net.lenni0451.commons.httpclient.requests.impl.PostRequest;
import net.raphimc.minecraftauth.MinecraftAuth;
import net.raphimc.minecraftauth.responsehandler.MinecraftResponseHandler;
import net.raphimc.minecraftauth.step.AbstractStep;
import net.raphimc.minecraftauth.util.CryptUtil;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Base64;

public class StepPlayerCertificates extends AbstractStep<StepMCToken.MCToken, StepPlayerCertificates.PlayerCertificates> {
    public static final String PLAYER_CERTIFICATES_URL = "https://api.minecraftservices.com/player/certificates";

    public StepPlayerCertificates(final AbstractStep<?, StepMCToken.MCToken> prevStep) {
        super("playerCertificates", prevStep);
    }

    @Override
    public PlayerCertificates applyStep(final HttpClient httpClient, final StepMCToken.MCToken mcToken) throws Exception {
        MinecraftAuth.LOGGER.info("Getting player certificates...");
        final PostRequest postRequest = new PostRequest(PLAYER_CERTIFICATES_URL);
        postRequest.setContent(new StringContent(ContentTypes.APPLICATION_JSON, ""));
        postRequest.setHeader(Headers.AUTHORIZATION, mcToken.getTokenType() + " " + mcToken.getAccessToken());
        final JsonObject obj = httpClient.execute(postRequest, new MinecraftResponseHandler());
        final JsonObject keyPair = obj.getAsJsonObject("keyPair");
        final PKCS8EncodedKeySpec encodedPrivateKey = new PKCS8EncodedKeySpec(Base64.getMimeDecoder().decode(keyPair.get("privateKey").getAsString().replace("-----BEGIN RSA PRIVATE KEY-----", "").replace("-----END RSA PRIVATE KEY-----", "")));
        final RSAPrivateKey privateKey = (RSAPrivateKey) CryptUtil.RSA_KEYFACTORY.generatePrivate(encodedPrivateKey);
        final X509EncodedKeySpec encodedPublicKey = new X509EncodedKeySpec(Base64.getMimeDecoder().decode(keyPair.get("publicKey").getAsString().replace("-----BEGIN RSA PUBLIC KEY-----", "").replace("-----END RSA PUBLIC KEY-----", "")));
        final RSAPublicKey publicKey = (RSAPublicKey) CryptUtil.RSA_KEYFACTORY.generatePublic(encodedPublicKey);
        final PlayerCertificates playerCertificates = new PlayerCertificates(Instant.parse(obj.get("expiresAt").getAsString()).toEpochMilli(), publicKey, privateKey, Base64.getMimeDecoder().decode(obj.get("publicKeySignatureV2").getAsString()), obj.has("publicKeySignature") ? Base64.getMimeDecoder().decode(obj.get("publicKeySignature").getAsString()) : new byte[0], mcToken);
        MinecraftAuth.LOGGER.info("Got player certificates, expires: " + Instant.ofEpochMilli(playerCertificates.getExpireTimeMs()).atZone(ZoneId.systemDefault()));
        return playerCertificates;
    }

    @Override
    public PlayerCertificates fromJson(final JsonObject json) {
        final StepMCToken.MCToken mcToken = this.prevStep != null ? this.prevStep.fromJson(json.getAsJsonObject(this.prevStep.name)) : null;
        return new PlayerCertificates(json.get("expireTimeMs").getAsLong(), CryptUtil.publicKeyRsaFromBase64(json.get("publicKey").getAsString()), CryptUtil.privateKeyRsaFromBase64(json.get("privateKey").getAsString()), Base64.getDecoder().decode(json.get("publicKeySignature").getAsString()), Base64.getDecoder().decode(json.get("legacyPublicKeySignature").getAsString()), mcToken);
    }

    @Override
    public JsonObject toJson(final PlayerCertificates playerCertificates) {
        final JsonObject json = new JsonObject();
        json.addProperty("expireTimeMs", playerCertificates.expireTimeMs);
        json.addProperty("publicKey", Base64.getEncoder().encodeToString(playerCertificates.publicKey.getEncoded()));
        json.addProperty("privateKey", Base64.getEncoder().encodeToString(playerCertificates.privateKey.getEncoded()));
        json.addProperty("publicKeySignature", Base64.getEncoder().encodeToString(playerCertificates.publicKeySignature));
        json.addProperty("legacyPublicKeySignature", Base64.getEncoder().encodeToString(playerCertificates.legacyPublicKeySignature));
        if (this.prevStep != null) json.add(this.prevStep.name, this.prevStep.toJson(playerCertificates.mcToken));
        return json;
    }


    public static final class PlayerCertificates extends AbstractStep.StepResult<StepMCToken.MCToken> {
        private final long expireTimeMs;
        private final RSAPublicKey publicKey;
        private final RSAPrivateKey privateKey;
        private final byte[] publicKeySignature;
        private final byte[] legacyPublicKeySignature;
        private final StepMCToken.MCToken mcToken;

        @Override
        protected StepMCToken.MCToken prevResult() {
            return this.mcToken;
        }

        @Override
        public boolean isExpired() {
            return this.expireTimeMs <= System.currentTimeMillis();
        }

        public PlayerCertificates(final long expireTimeMs, final RSAPublicKey publicKey, final RSAPrivateKey privateKey, final byte[] publicKeySignature, final byte[] legacyPublicKeySignature, final StepMCToken.MCToken mcToken) {
            this.expireTimeMs = expireTimeMs;
            this.publicKey = publicKey;
            this.privateKey = privateKey;
            this.publicKeySignature = publicKeySignature;
            this.legacyPublicKeySignature = legacyPublicKeySignature;
            this.mcToken = mcToken;
        }

        public long getExpireTimeMs() {
            return this.expireTimeMs;
        }

        public RSAPublicKey getPublicKey() {
            return this.publicKey;
        }

        public RSAPrivateKey getPrivateKey() {
            return this.privateKey;
        }

        public byte[] getPublicKeySignature() {
            return this.publicKeySignature;
        }

        public byte[] getLegacyPublicKeySignature() {
            return this.legacyPublicKeySignature;
        }

        public StepMCToken.MCToken getMcToken() {
            return this.mcToken;
        }

        @Override
        public String toString() {
            return "StepPlayerCertificates.PlayerCertificates(expireTimeMs=" + this.getExpireTimeMs() + ", publicKey=" + this.getPublicKey() + ", privateKey=" + this.getPrivateKey() + ", publicKeySignature=" + java.util.Arrays.toString(this.getPublicKeySignature()) + ", legacyPublicKeySignature=" + java.util.Arrays.toString(this.getLegacyPublicKeySignature()) + ", mcToken=" + this.getMcToken() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepPlayerCertificates.PlayerCertificates)) return false;
            final StepPlayerCertificates.PlayerCertificates other = (StepPlayerCertificates.PlayerCertificates) o;
            if (!other.canEqual((Object) this)) return false;
            if (this.getExpireTimeMs() != other.getExpireTimeMs()) return false;
            final Object this$publicKey = this.getPublicKey();
            final Object other$publicKey = other.getPublicKey();
            if (this$publicKey == null ? other$publicKey != null : !this$publicKey.equals(other$publicKey)) return false;
            final Object this$privateKey = this.getPrivateKey();
            final Object other$privateKey = other.getPrivateKey();
            if (this$privateKey == null ? other$privateKey != null : !this$privateKey.equals(other$privateKey)) return false;
            if (!java.util.Arrays.equals(this.getPublicKeySignature(), other.getPublicKeySignature())) return false;
            if (!java.util.Arrays.equals(this.getLegacyPublicKeySignature(), other.getLegacyPublicKeySignature())) return false;
            final Object this$mcToken = this.getMcToken();
            final Object other$mcToken = other.getMcToken();
            if (this$mcToken == null ? other$mcToken != null : !this$mcToken.equals(other$mcToken)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepPlayerCertificates.PlayerCertificates;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final long $expireTimeMs = this.getExpireTimeMs();
            result = result * PRIME + (int) ($expireTimeMs >>> 32 ^ $expireTimeMs);
            final Object $publicKey = this.getPublicKey();
            result = result * PRIME + ($publicKey == null ? 43 : $publicKey.hashCode());
            final Object $privateKey = this.getPrivateKey();
            result = result * PRIME + ($privateKey == null ? 43 : $privateKey.hashCode());
            result = result * PRIME + java.util.Arrays.hashCode(this.getPublicKeySignature());
            result = result * PRIME + java.util.Arrays.hashCode(this.getLegacyPublicKeySignature());
            final Object $mcToken = this.getMcToken();
            result = result * PRIME + ($mcToken == null ? 43 : $mcToken.hashCode());
            return result;
        }
    }
}
