/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.msa;

import com.google.gson.JsonObject;
import net.raphimc.minecraftauth.step.AbstractStep;
import net.raphimc.minecraftauth.util.JsonUtil;
import net.raphimc.minecraftauth.util.OAuthEnvironment;
import net.raphimc.minecraftauth.util.UuidUtil;
import java.util.HashMap;
import java.util.Map;

public abstract class MsaCodeStep<I extends AbstractStep.StepResult<?>> extends AbstractStep<I, MsaCodeStep.MsaCode> {
    public MsaCodeStep(final AbstractStep<?, I> prevStep) {
        super("msaCode", prevStep);
    }

    @Override
    public MsaCode fromJson(final JsonObject json) {
        return new MsaCode(json.get("code").getAsString(), new ApplicationDetails(json.get("clientId").getAsString(), json.get("scope").getAsString(), JsonUtil.getStringOr(json, "clientSecret", null), JsonUtil.getStringOr(json, "redirectUri", null), OAuthEnvironment.valueOf(JsonUtil.getStringOr(json, "oAuthEnvironment", "LIVE"))));
    }

    @Override
    public JsonObject toJson(final MsaCode msaCode) {
        final JsonObject json = new JsonObject();
        json.addProperty("code", msaCode.code);
        json.addProperty("clientId", msaCode.applicationDetails.clientId);
        json.addProperty("scope", msaCode.applicationDetails.scope);
        json.addProperty("clientSecret", msaCode.applicationDetails.clientSecret);
        json.addProperty("redirectUri", msaCode.applicationDetails.redirectUri);
        json.addProperty("oAuthEnvironment", msaCode.applicationDetails.oAuthEnvironment.name());
        return json;
    }


    public static final class ApplicationDetails extends AbstractStep.FirstStepResult {
        private final String clientId;
        private final String scope;
        private final String clientSecret;
        private final String redirectUri;
        private final OAuthEnvironment oAuthEnvironment;

        public boolean isTitleClientId() {
            return !UuidUtil.isDashedUuid(this.clientId);
        }

        public Map<String, String> getOAuthParameters() {
            final Map<String, String> parameters = new HashMap<>();
            parameters.put("client_id", this.clientId);
            parameters.put("scope", this.scope);
            parameters.put("redirect_uri", this.redirectUri);
            parameters.put("response_type", "code");
            parameters.put("response_mode", "query");
            return parameters;
        }

        public ApplicationDetails(final String clientId, final String scope, final String clientSecret, final String redirectUri, final OAuthEnvironment oAuthEnvironment) {
            this.clientId = clientId;
            this.scope = scope;
            this.clientSecret = clientSecret;
            this.redirectUri = redirectUri;
            this.oAuthEnvironment = oAuthEnvironment;
        }

        public String getClientId() {
            return this.clientId;
        }

        public String getScope() {
            return this.scope;
        }

        public String getClientSecret() {
            return this.clientSecret;
        }

        public String getRedirectUri() {
            return this.redirectUri;
        }

        public OAuthEnvironment getOAuthEnvironment() {
            return this.oAuthEnvironment;
        }

        @Override
        public String toString() {
            return "MsaCodeStep.ApplicationDetails(clientId=" + this.getClientId() + ", scope=" + this.getScope() + ", clientSecret=" + this.getClientSecret() + ", redirectUri=" + this.getRedirectUri() + ", oAuthEnvironment=" + this.getOAuthEnvironment() + ")";
        }

        /**
         * @return a clone of this object, except with this updated property (returns {@code this} if an identical value is passed).
         */
        public MsaCodeStep.ApplicationDetails withClientId(final String clientId) {
            return this.clientId == clientId ? this : new MsaCodeStep.ApplicationDetails(clientId, this.scope, this.clientSecret, this.redirectUri, this.oAuthEnvironment);
        }

        /**
         * @return a clone of this object, except with this updated property (returns {@code this} if an identical value is passed).
         */
        public MsaCodeStep.ApplicationDetails withScope(final String scope) {
            return this.scope == scope ? this : new MsaCodeStep.ApplicationDetails(this.clientId, scope, this.clientSecret, this.redirectUri, this.oAuthEnvironment);
        }

        /**
         * @return a clone of this object, except with this updated property (returns {@code this} if an identical value is passed).
         */
        public MsaCodeStep.ApplicationDetails withClientSecret(final String clientSecret) {
            return this.clientSecret == clientSecret ? this : new MsaCodeStep.ApplicationDetails(this.clientId, this.scope, clientSecret, this.redirectUri, this.oAuthEnvironment);
        }

        /**
         * @return a clone of this object, except with this updated property (returns {@code this} if an identical value is passed).
         */
        public MsaCodeStep.ApplicationDetails withRedirectUri(final String redirectUri) {
            return this.redirectUri == redirectUri ? this : new MsaCodeStep.ApplicationDetails(this.clientId, this.scope, this.clientSecret, redirectUri, this.oAuthEnvironment);
        }

        /**
         * @return a clone of this object, except with this updated property (returns {@code this} if an identical value is passed).
         */
        public MsaCodeStep.ApplicationDetails withOAuthEnvironment(final OAuthEnvironment oAuthEnvironment) {
            return this.oAuthEnvironment == oAuthEnvironment ? this : new MsaCodeStep.ApplicationDetails(this.clientId, this.scope, this.clientSecret, this.redirectUri, oAuthEnvironment);
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof MsaCodeStep.ApplicationDetails)) return false;
            final MsaCodeStep.ApplicationDetails other = (MsaCodeStep.ApplicationDetails) o;
            if (!other.canEqual((Object) this)) return false;
            final Object this$clientId = this.getClientId();
            final Object other$clientId = other.getClientId();
            if (this$clientId == null ? other$clientId != null : !this$clientId.equals(other$clientId)) return false;
            final Object this$scope = this.getScope();
            final Object other$scope = other.getScope();
            if (this$scope == null ? other$scope != null : !this$scope.equals(other$scope)) return false;
            final Object this$clientSecret = this.getClientSecret();
            final Object other$clientSecret = other.getClientSecret();
            if (this$clientSecret == null ? other$clientSecret != null : !this$clientSecret.equals(other$clientSecret)) return false;
            final Object this$redirectUri = this.getRedirectUri();
            final Object other$redirectUri = other.getRedirectUri();
            if (this$redirectUri == null ? other$redirectUri != null : !this$redirectUri.equals(other$redirectUri)) return false;
            final Object this$oAuthEnvironment = this.getOAuthEnvironment();
            final Object other$oAuthEnvironment = other.getOAuthEnvironment();
            if (this$oAuthEnvironment == null ? other$oAuthEnvironment != null : !this$oAuthEnvironment.equals(other$oAuthEnvironment)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof MsaCodeStep.ApplicationDetails;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final Object $clientId = this.getClientId();
            result = result * PRIME + ($clientId == null ? 43 : $clientId.hashCode());
            final Object $scope = this.getScope();
            result = result * PRIME + ($scope == null ? 43 : $scope.hashCode());
            final Object $clientSecret = this.getClientSecret();
            result = result * PRIME + ($clientSecret == null ? 43 : $clientSecret.hashCode());
            final Object $redirectUri = this.getRedirectUri();
            result = result * PRIME + ($redirectUri == null ? 43 : $redirectUri.hashCode());
            final Object $oAuthEnvironment = this.getOAuthEnvironment();
            result = result * PRIME + ($oAuthEnvironment == null ? 43 : $oAuthEnvironment.hashCode());
            return result;
        }
    }


    public static final class MsaCode extends AbstractStep.StepResult<ApplicationDetails> {
        private final String code;
        private final ApplicationDetails applicationDetails;

        @Override
        protected ApplicationDetails prevResult() {
            return this.applicationDetails;
        }

        public MsaCode(final String code, final ApplicationDetails applicationDetails) {
            this.code = code;
            this.applicationDetails = applicationDetails;
        }

        public String getCode() {
            return this.code;
        }

        public ApplicationDetails getApplicationDetails() {
            return this.applicationDetails;
        }

        @Override
        public String toString() {
            return "MsaCodeStep.MsaCode(code=" + this.getCode() + ", applicationDetails=" + this.getApplicationDetails() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof MsaCodeStep.MsaCode)) return false;
            final MsaCodeStep.MsaCode other = (MsaCodeStep.MsaCode) o;
            if (!other.canEqual((Object) this)) return false;
            final Object this$code = this.getCode();
            final Object other$code = other.getCode();
            if (this$code == null ? other$code != null : !this$code.equals(other$code)) return false;
            final Object this$applicationDetails = this.getApplicationDetails();
            final Object other$applicationDetails = other.getApplicationDetails();
            if (this$applicationDetails == null ? other$applicationDetails != null : !this$applicationDetails.equals(other$applicationDetails)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof MsaCodeStep.MsaCode;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final Object $code = this.getCode();
            result = result * PRIME + ($code == null ? 43 : $code.hashCode());
            final Object $applicationDetails = this.getApplicationDetails();
            result = result * PRIME + ($applicationDetails == null ? 43 : $applicationDetails.hashCode());
            return result;
        }
    }
}
