/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.msa;

import com.google.gson.JsonObject;
import net.lenni0451.commons.httpclient.HttpClient;
import net.lenni0451.commons.httpclient.utils.URLWrapper;
import net.raphimc.minecraftauth.MinecraftAuth;
import net.raphimc.minecraftauth.step.AbstractStep;
import java.net.ServerSocket;
import java.net.URL;
import java.util.function.Consumer;

public class StepLocalWebServer extends AbstractStep<StepLocalWebServer.LocalWebServerCallback, StepLocalWebServer.LocalWebServer> {
    private final MsaCodeStep.ApplicationDetails applicationDetails;

    public StepLocalWebServer(final MsaCodeStep.ApplicationDetails applicationDetails) {
        super("localWebServer", null);
        if (applicationDetails.getRedirectUri().endsWith("/")) {
            throw new IllegalArgumentException("Redirect URI must not end with a slash");
        }
        this.applicationDetails = applicationDetails;
    }

    @Override
    public LocalWebServer applyStep(final HttpClient httpClient, final LocalWebServerCallback localWebServerCallback) throws Exception {
        MinecraftAuth.LOGGER.info("Creating URL for MSA login via local webserver...");
        if (localWebServerCallback == null) {
            throw new IllegalStateException("Missing StepLocalWebServer.LocalWebServerCallback input");
        }
        try (ServerSocket localServer = new ServerSocket(0)) {
            final int localPort = localServer.getLocalPort();
            final URL authenticationUrl = new URLWrapper(this.applicationDetails.getOAuthEnvironment().getAuthorizeUrl()).wrapQuery().addQueries(this.applicationDetails.getOAuthParameters()).setQuery("redirect_uri", this.applicationDetails.getRedirectUri() + ":" + localPort).setQuery("prompt", "select_account").apply().toURL();
            final LocalWebServer localWebServer = new LocalWebServer(authenticationUrl.toString(), localPort, this.applicationDetails);
            MinecraftAuth.LOGGER.info("Created local webserver MSA authentication URL: " + localWebServer.getAuthenticationUrl());
            localWebServerCallback.callback.accept(localWebServer);
            return localWebServer;
        }
    }

    @Override
    public LocalWebServer fromJson(final JsonObject json) {
        return new LocalWebServer(json.get("authenticationUrl").getAsString(), json.get("port").getAsInt(), this.applicationDetails);
    }

    @Override
    public JsonObject toJson(final LocalWebServer localWebServer) {
        final JsonObject json = new JsonObject();
        json.addProperty("authenticationUrl", localWebServer.authenticationUrl);
        json.addProperty("port", localWebServer.port);
        return json;
    }


    public static final class LocalWebServer extends AbstractStep.StepResult<MsaCodeStep.ApplicationDetails> {
        private final String authenticationUrl;
        private final int port;
        private final MsaCodeStep.ApplicationDetails applicationDetails;

        public LocalWebServer(final String authenticationUrl, final int port, final MsaCodeStep.ApplicationDetails applicationDetails) {
            this.authenticationUrl = authenticationUrl;
            this.port = port;
            this.applicationDetails = applicationDetails.withRedirectUri(applicationDetails.getRedirectUri() + ":" + port);
        }

        @Override
        protected MsaCodeStep.ApplicationDetails prevResult() {
            return this.applicationDetails;
        }

        public String getAuthenticationUrl() {
            return this.authenticationUrl;
        }

        public int getPort() {
            return this.port;
        }

        public MsaCodeStep.ApplicationDetails getApplicationDetails() {
            return this.applicationDetails;
        }

        @Override
        public String toString() {
            return "StepLocalWebServer.LocalWebServer(authenticationUrl=" + this.getAuthenticationUrl() + ", port=" + this.getPort() + ", applicationDetails=" + this.getApplicationDetails() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepLocalWebServer.LocalWebServer)) return false;
            final StepLocalWebServer.LocalWebServer other = (StepLocalWebServer.LocalWebServer) o;
            if (!other.canEqual((Object) this)) return false;
            if (this.getPort() != other.getPort()) return false;
            final Object this$authenticationUrl = this.getAuthenticationUrl();
            final Object other$authenticationUrl = other.getAuthenticationUrl();
            if (this$authenticationUrl == null ? other$authenticationUrl != null : !this$authenticationUrl.equals(other$authenticationUrl)) return false;
            final Object this$applicationDetails = this.getApplicationDetails();
            final Object other$applicationDetails = other.getApplicationDetails();
            if (this$applicationDetails == null ? other$applicationDetails != null : !this$applicationDetails.equals(other$applicationDetails)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepLocalWebServer.LocalWebServer;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            result = result * PRIME + this.getPort();
            final Object $authenticationUrl = this.getAuthenticationUrl();
            result = result * PRIME + ($authenticationUrl == null ? 43 : $authenticationUrl.hashCode());
            final Object $applicationDetails = this.getApplicationDetails();
            result = result * PRIME + ($applicationDetails == null ? 43 : $applicationDetails.hashCode());
            return result;
        }
    }


    public static final class LocalWebServerCallback extends AbstractStep.InitialInput {
        private final Consumer<LocalWebServer> callback;

        public LocalWebServerCallback(final Consumer<LocalWebServer> callback) {
            this.callback = callback;
        }

        public Consumer<LocalWebServer> getCallback() {
            return this.callback;
        }

        @Override
        public String toString() {
            return "StepLocalWebServer.LocalWebServerCallback(callback=" + this.getCallback() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepLocalWebServer.LocalWebServerCallback)) return false;
            final StepLocalWebServer.LocalWebServerCallback other = (StepLocalWebServer.LocalWebServerCallback) o;
            if (!other.canEqual((Object) this)) return false;
            final Object this$callback = this.getCallback();
            final Object other$callback = other.getCallback();
            if (this$callback == null ? other$callback != null : !this$callback.equals(other$callback)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepLocalWebServer.LocalWebServerCallback;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final Object $callback = this.getCallback();
            result = result * PRIME + ($callback == null ? 43 : $callback.hashCode());
            return result;
        }
    }
}
