/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.msa;

import com.google.gson.JsonObject;
import net.lenni0451.commons.httpclient.HttpClient;
import net.lenni0451.commons.httpclient.content.impl.URLEncodedFormContent;
import net.lenni0451.commons.httpclient.requests.impl.PostRequest;
import net.raphimc.minecraftauth.MinecraftAuth;
import net.raphimc.minecraftauth.responsehandler.MsaResponseHandler;
import net.raphimc.minecraftauth.step.AbstractStep;
import net.raphimc.minecraftauth.util.OAuthEnvironment;
import java.time.Instant;
import java.time.ZoneId;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;

public class StepMsaDeviceCode extends AbstractStep<StepMsaDeviceCode.MsaDeviceCodeCallback, StepMsaDeviceCode.MsaDeviceCode> {
    private final MsaCodeStep.ApplicationDetails applicationDetails;

    public StepMsaDeviceCode(final MsaCodeStep.ApplicationDetails applicationDetails) {
        super("msaDeviceCode", null);
        this.applicationDetails = applicationDetails;
    }

    @Override
    public MsaDeviceCode applyStep(final HttpClient httpClient, final MsaDeviceCodeCallback msaDeviceCodeCallback) throws Exception {
        MinecraftAuth.LOGGER.info("Getting device code for MSA login...");
        if (msaDeviceCodeCallback == null) {
            throw new IllegalStateException("Missing StepMsaDeviceCode.MsaDeviceCodeCallback input");
        }
        final Map<String, String> postData = new HashMap<>();
        postData.put("client_id", this.applicationDetails.getClientId());
        postData.put("scope", this.applicationDetails.getScope());
        if (this.applicationDetails.getOAuthEnvironment() == OAuthEnvironment.LIVE) {
            postData.put("response_type", "device_code");
        }
        final PostRequest postRequest = new PostRequest(this.applicationDetails.getOAuthEnvironment().getDeviceCodeUrl());
        postRequest.setContent(new URLEncodedFormContent(postData));
        final JsonObject obj = httpClient.execute(postRequest, new MsaResponseHandler());
        final MsaDeviceCode msaDeviceCode = new MsaDeviceCode(System.currentTimeMillis() + obj.get("expires_in").getAsLong() * 1000, obj.get("interval").getAsLong() * 1000, obj.get("device_code").getAsString(), obj.get("user_code").getAsString(), obj.get("verification_uri").getAsString(), this.applicationDetails);
        MinecraftAuth.LOGGER.info("Got MSA device code, expires: " + Instant.ofEpochMilli(msaDeviceCode.getExpireTimeMs()).atZone(ZoneId.systemDefault()));
        msaDeviceCodeCallback.callback.accept(msaDeviceCode);
        return msaDeviceCode;
    }

    @Override
    public MsaDeviceCode fromJson(final JsonObject json) {
        return new MsaDeviceCode(json.get("expireTimeMs").getAsLong(), json.get("intervalMs").getAsLong(), json.get("deviceCode").getAsString(), json.get("userCode").getAsString(), json.get("verificationUrl").getAsString(), this.applicationDetails);
    }

    @Override
    public JsonObject toJson(final MsaDeviceCode msaDeviceCode) {
        final JsonObject json = new JsonObject();
        json.addProperty("expireTimeMs", msaDeviceCode.expireTimeMs);
        json.addProperty("intervalMs", msaDeviceCode.intervalMs);
        json.addProperty("deviceCode", msaDeviceCode.deviceCode);
        json.addProperty("userCode", msaDeviceCode.userCode);
        json.addProperty("verificationUrl", msaDeviceCode.verificationUri);
        return json;
    }


    public static final class MsaDeviceCode extends AbstractStep.StepResult<MsaCodeStep.ApplicationDetails> {
        private final long expireTimeMs;
        private final long intervalMs;
        private final String deviceCode;
        private final String userCode;
        private final String verificationUri;
        private final MsaCodeStep.ApplicationDetails applicationDetails;

        @Override
        protected MsaCodeStep.ApplicationDetails prevResult() {
            return this.applicationDetails;
        }

        @Override
        public boolean isExpired() {
            return this.expireTimeMs <= System.currentTimeMillis();
        }

        public String getDirectVerificationUri() {
            return this.verificationUri + "?otc=" + this.userCode;
        }

        public MsaDeviceCode(final long expireTimeMs, final long intervalMs, final String deviceCode, final String userCode, final String verificationUri, final MsaCodeStep.ApplicationDetails applicationDetails) {
            this.expireTimeMs = expireTimeMs;
            this.intervalMs = intervalMs;
            this.deviceCode = deviceCode;
            this.userCode = userCode;
            this.verificationUri = verificationUri;
            this.applicationDetails = applicationDetails;
        }

        public long getExpireTimeMs() {
            return this.expireTimeMs;
        }

        public long getIntervalMs() {
            return this.intervalMs;
        }

        public String getDeviceCode() {
            return this.deviceCode;
        }

        public String getUserCode() {
            return this.userCode;
        }

        public String getVerificationUri() {
            return this.verificationUri;
        }

        public MsaCodeStep.ApplicationDetails getApplicationDetails() {
            return this.applicationDetails;
        }

        @Override
        public String toString() {
            return "StepMsaDeviceCode.MsaDeviceCode(expireTimeMs=" + this.getExpireTimeMs() + ", intervalMs=" + this.getIntervalMs() + ", deviceCode=" + this.getDeviceCode() + ", userCode=" + this.getUserCode() + ", verificationUri=" + this.getVerificationUri() + ", applicationDetails=" + this.getApplicationDetails() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepMsaDeviceCode.MsaDeviceCode)) return false;
            final StepMsaDeviceCode.MsaDeviceCode other = (StepMsaDeviceCode.MsaDeviceCode) o;
            if (!other.canEqual((Object) this)) return false;
            if (this.getExpireTimeMs() != other.getExpireTimeMs()) return false;
            if (this.getIntervalMs() != other.getIntervalMs()) return false;
            final Object this$deviceCode = this.getDeviceCode();
            final Object other$deviceCode = other.getDeviceCode();
            if (this$deviceCode == null ? other$deviceCode != null : !this$deviceCode.equals(other$deviceCode)) return false;
            final Object this$userCode = this.getUserCode();
            final Object other$userCode = other.getUserCode();
            if (this$userCode == null ? other$userCode != null : !this$userCode.equals(other$userCode)) return false;
            final Object this$verificationUri = this.getVerificationUri();
            final Object other$verificationUri = other.getVerificationUri();
            if (this$verificationUri == null ? other$verificationUri != null : !this$verificationUri.equals(other$verificationUri)) return false;
            final Object this$applicationDetails = this.getApplicationDetails();
            final Object other$applicationDetails = other.getApplicationDetails();
            if (this$applicationDetails == null ? other$applicationDetails != null : !this$applicationDetails.equals(other$applicationDetails)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepMsaDeviceCode.MsaDeviceCode;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final long $expireTimeMs = this.getExpireTimeMs();
            result = result * PRIME + (int) ($expireTimeMs >>> 32 ^ $expireTimeMs);
            final long $intervalMs = this.getIntervalMs();
            result = result * PRIME + (int) ($intervalMs >>> 32 ^ $intervalMs);
            final Object $deviceCode = this.getDeviceCode();
            result = result * PRIME + ($deviceCode == null ? 43 : $deviceCode.hashCode());
            final Object $userCode = this.getUserCode();
            result = result * PRIME + ($userCode == null ? 43 : $userCode.hashCode());
            final Object $verificationUri = this.getVerificationUri();
            result = result * PRIME + ($verificationUri == null ? 43 : $verificationUri.hashCode());
            final Object $applicationDetails = this.getApplicationDetails();
            result = result * PRIME + ($applicationDetails == null ? 43 : $applicationDetails.hashCode());
            return result;
        }
    }


    public static final class MsaDeviceCodeCallback extends AbstractStep.InitialInput {
        private final Consumer<MsaDeviceCode> callback;

        public MsaDeviceCodeCallback(final Consumer<MsaDeviceCode> callback) {
            this.callback = callback;
        }

        public Consumer<MsaDeviceCode> getCallback() {
            return this.callback;
        }

        @Override
        public String toString() {
            return "StepMsaDeviceCode.MsaDeviceCodeCallback(callback=" + this.getCallback() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepMsaDeviceCode.MsaDeviceCodeCallback)) return false;
            final StepMsaDeviceCode.MsaDeviceCodeCallback other = (StepMsaDeviceCode.MsaDeviceCodeCallback) o;
            if (!other.canEqual((Object) this)) return false;
            final Object this$callback = this.getCallback();
            final Object other$callback = other.getCallback();
            if (this$callback == null ? other$callback != null : !this$callback.equals(other$callback)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepMsaDeviceCode.MsaDeviceCodeCallback;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final Object $callback = this.getCallback();
            result = result * PRIME + ($callback == null ? 43 : $callback.hashCode());
            return result;
        }
    }
}
