/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.xbl;

import com.google.gson.JsonObject;
import net.lenni0451.commons.httpclient.HttpClient;
import net.lenni0451.commons.httpclient.requests.impl.PostRequest;
import net.raphimc.minecraftauth.MinecraftAuth;
import net.raphimc.minecraftauth.responsehandler.XblResponseHandler;
import net.raphimc.minecraftauth.step.AbstractStep;
import net.raphimc.minecraftauth.util.CryptUtil;
import net.raphimc.minecraftauth.util.JsonContent;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECGenParameterSpec;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Base64;
import java.util.UUID;

public class StepXblDeviceToken extends AbstractStep<AbstractStep.StepResult<?>, StepXblDeviceToken.XblDeviceToken> {
    public static final String XBL_DEVICE_URL = "https://device.auth.xboxlive.com/device/authenticate";
    private final String deviceType;

    public StepXblDeviceToken(final String deviceType) {
        super("xblDeviceToken", null);
        this.deviceType = deviceType;
    }

    @Override
    public XblDeviceToken applyStep(final HttpClient httpClient, final StepResult<?> prevResult) throws Exception {
        MinecraftAuth.LOGGER.info("Authenticating device with Xbox Live...");
        final UUID id = UUID.randomUUID();
        final KeyPairGenerator secp256r1 = KeyPairGenerator.getInstance("EC");
        secp256r1.initialize(new ECGenParameterSpec("secp256r1"));
        final KeyPair ecdsa256KeyPair = secp256r1.generateKeyPair();
        final ECPublicKey publicKey = (ECPublicKey) ecdsa256KeyPair.getPublic();
        final ECPrivateKey privateKey = (ECPrivateKey) ecdsa256KeyPair.getPrivate();
        final JsonObject postData = new JsonObject();
        final JsonObject properties = new JsonObject();
        properties.addProperty("AuthMethod", "ProofOfPossession");
        properties.addProperty("DeviceType", this.deviceType);
        properties.addProperty("Id", "{" + id + "}");
        properties.add("ProofKey", CryptUtil.getProofKey(publicKey));
        properties.addProperty("Version", "0.0.0");
        postData.add("Properties", properties);
        postData.addProperty("RelyingParty", "http://auth.xboxlive.com");
        postData.addProperty("TokenType", "JWT");
        final PostRequest postRequest = new PostRequest(XBL_DEVICE_URL);
        postRequest.setContent(new JsonContent(postData));
        postRequest.setHeader("x-xbl-contract-version", "1");
        postRequest.setHeader(CryptUtil.getSignatureHeader(postRequest, privateKey));
        final JsonObject obj = httpClient.execute(postRequest, new XblResponseHandler());
        final XblDeviceToken xblDeviceToken = new XblDeviceToken(publicKey, privateKey, id, Instant.parse(obj.get("NotAfter").getAsString()).toEpochMilli(), obj.get("Token").getAsString(), obj.getAsJsonObject("DisplayClaims").getAsJsonObject("xdi").get("did").getAsString());
        MinecraftAuth.LOGGER.info("Got XBL Device Token, expires: " + Instant.ofEpochMilli(xblDeviceToken.getExpireTimeMs()).atZone(ZoneId.systemDefault()));
        return xblDeviceToken;
    }

    @Override
    public XblDeviceToken fromJson(final JsonObject json) {
        return new XblDeviceToken(CryptUtil.publicKeyEcFromBase64(json.get("publicKey").getAsString()), CryptUtil.privateKeyEcFromBase64(json.get("privateKey").getAsString()), UUID.fromString(json.get("id").getAsString()), json.get("expireTimeMs").getAsLong(), json.get("token").getAsString(), json.get("deviceId").getAsString());
    }

    @Override
    public JsonObject toJson(final XblDeviceToken xblDeviceToken) {
        final JsonObject json = new JsonObject();
        json.addProperty("publicKey", Base64.getEncoder().encodeToString(xblDeviceToken.publicKey.getEncoded()));
        json.addProperty("privateKey", Base64.getEncoder().encodeToString(xblDeviceToken.privateKey.getEncoded()));
        json.addProperty("id", xblDeviceToken.id.toString());
        json.addProperty("expireTimeMs", xblDeviceToken.expireTimeMs);
        json.addProperty("token", xblDeviceToken.token);
        json.addProperty("deviceId", xblDeviceToken.deviceId);
        return json;
    }


    public static final class XblDeviceToken extends AbstractStep.FirstStepResult {
        private final ECPublicKey publicKey;
        private final ECPrivateKey privateKey;
        private final UUID id;
        private final long expireTimeMs;
        private final String token;
        private final String deviceId;

        @Override
        public boolean isExpired() {
            return this.expireTimeMs <= System.currentTimeMillis();
        }

        public XblDeviceToken(final ECPublicKey publicKey, final ECPrivateKey privateKey, final UUID id, final long expireTimeMs, final String token, final String deviceId) {
            this.publicKey = publicKey;
            this.privateKey = privateKey;
            this.id = id;
            this.expireTimeMs = expireTimeMs;
            this.token = token;
            this.deviceId = deviceId;
        }

        public ECPublicKey getPublicKey() {
            return this.publicKey;
        }

        public ECPrivateKey getPrivateKey() {
            return this.privateKey;
        }

        public UUID getId() {
            return this.id;
        }

        public long getExpireTimeMs() {
            return this.expireTimeMs;
        }

        public String getToken() {
            return this.token;
        }

        public String getDeviceId() {
            return this.deviceId;
        }

        @Override
        public String toString() {
            return "StepXblDeviceToken.XblDeviceToken(publicKey=" + this.getPublicKey() + ", privateKey=" + this.getPrivateKey() + ", id=" + this.getId() + ", expireTimeMs=" + this.getExpireTimeMs() + ", token=" + this.getToken() + ", deviceId=" + this.getDeviceId() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepXblDeviceToken.XblDeviceToken)) return false;
            final StepXblDeviceToken.XblDeviceToken other = (StepXblDeviceToken.XblDeviceToken) o;
            if (!other.canEqual((Object) this)) return false;
            if (this.getExpireTimeMs() != other.getExpireTimeMs()) return false;
            final Object this$publicKey = this.getPublicKey();
            final Object other$publicKey = other.getPublicKey();
            if (this$publicKey == null ? other$publicKey != null : !this$publicKey.equals(other$publicKey)) return false;
            final Object this$privateKey = this.getPrivateKey();
            final Object other$privateKey = other.getPrivateKey();
            if (this$privateKey == null ? other$privateKey != null : !this$privateKey.equals(other$privateKey)) return false;
            final Object this$id = this.getId();
            final Object other$id = other.getId();
            if (this$id == null ? other$id != null : !this$id.equals(other$id)) return false;
            final Object this$token = this.getToken();
            final Object other$token = other.getToken();
            if (this$token == null ? other$token != null : !this$token.equals(other$token)) return false;
            final Object this$deviceId = this.getDeviceId();
            final Object other$deviceId = other.getDeviceId();
            if (this$deviceId == null ? other$deviceId != null : !this$deviceId.equals(other$deviceId)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepXblDeviceToken.XblDeviceToken;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final long $expireTimeMs = this.getExpireTimeMs();
            result = result * PRIME + (int) ($expireTimeMs >>> 32 ^ $expireTimeMs);
            final Object $publicKey = this.getPublicKey();
            result = result * PRIME + ($publicKey == null ? 43 : $publicKey.hashCode());
            final Object $privateKey = this.getPrivateKey();
            result = result * PRIME + ($privateKey == null ? 43 : $privateKey.hashCode());
            final Object $id = this.getId();
            result = result * PRIME + ($id == null ? 43 : $id.hashCode());
            final Object $token = this.getToken();
            result = result * PRIME + ($token == null ? 43 : $token.hashCode());
            final Object $deviceId = this.getDeviceId();
            result = result * PRIME + ($deviceId == null ? 43 : $deviceId.hashCode());
            return result;
        }
    }
}
