/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.xbl;

import com.google.gson.JsonObject;
import net.lenni0451.commons.httpclient.HttpClient;
import net.lenni0451.commons.httpclient.requests.impl.PostRequest;
import net.raphimc.minecraftauth.MinecraftAuth;
import net.raphimc.minecraftauth.responsehandler.XblResponseHandler;
import net.raphimc.minecraftauth.step.AbstractStep;
import net.raphimc.minecraftauth.step.xbl.session.StepFullXblSession;
import net.raphimc.minecraftauth.step.xbl.session.StepInitialXblSession;
import net.raphimc.minecraftauth.util.CryptUtil;
import net.raphimc.minecraftauth.util.JsonContent;
import java.time.Instant;
import java.time.ZoneId;

public class StepXblSisuAuthentication extends AbstractStep<StepInitialXblSession.InitialXblSession, StepXblSisuAuthentication.XblSisuTokens> {
    public static final String XBL_SISU_URL = "https://sisu.xboxlive.com/authorize";
    private final String relyingParty;

    public StepXblSisuAuthentication(final AbstractStep<?, StepInitialXblSession.InitialXblSession> prevStep, final String relyingParty) {
        super("xblSisuAuthentication", prevStep);
        this.relyingParty = relyingParty;
    }

    @Override
    public StepXblSisuAuthentication.XblSisuTokens applyStep(final HttpClient httpClient, final StepInitialXblSession.InitialXblSession initialXblSession) throws Exception {
        MinecraftAuth.LOGGER.info("Authenticating with Xbox Live using SISU...");
        if (initialXblSession.getXblDeviceToken() == null) {
            throw new IllegalStateException("An XBL Device Token is needed for SISU authentication");
        }
        if (!initialXblSession.getMsaToken().getMsaCode().getApplicationDetails().isTitleClientId()) {
            throw new IllegalStateException("A Title Client ID is needed for SISU authentication");
        }
        final JsonObject postData = new JsonObject();
        postData.addProperty("AccessToken", "t=" + initialXblSession.getMsaToken().getAccessToken());
        postData.addProperty("DeviceToken", initialXblSession.getXblDeviceToken().getToken());
        postData.addProperty("AppId", initialXblSession.getMsaToken().getMsaCode().getApplicationDetails().getClientId());
        postData.add("ProofKey", CryptUtil.getProofKey(initialXblSession.getXblDeviceToken().getPublicKey()));
        postData.addProperty("SiteName", "user.auth.xboxlive.com");
        postData.addProperty("RelyingParty", this.relyingParty);
        postData.addProperty("Sandbox", "RETAIL");
        postData.addProperty("UseModernGamertag", true);
        final PostRequest postRequest = new PostRequest(XBL_SISU_URL);
        postRequest.setContent(new JsonContent(postData));
        postRequest.setHeader(CryptUtil.getSignatureHeader(postRequest, initialXblSession.getXblDeviceToken().getPrivateKey()));
        final JsonObject obj = httpClient.execute(postRequest, new XblResponseHandler());
        final XblSisuTokens xblSisuTokens = new XblSisuTokens(new XblSisuTokens.SisuTitleToken(Instant.parse(obj.getAsJsonObject("TitleToken").get("NotAfter").getAsString()).toEpochMilli(), obj.getAsJsonObject("TitleToken").get("Token").getAsString(), obj.getAsJsonObject("TitleToken").getAsJsonObject("DisplayClaims").getAsJsonObject("xti").get("tid").getAsString()), new XblSisuTokens.SisuUserToken(Instant.parse(obj.getAsJsonObject("UserToken").get("NotAfter").getAsString()).toEpochMilli(), obj.getAsJsonObject("UserToken").get("Token").getAsString(), obj.getAsJsonObject("UserToken").getAsJsonObject("DisplayClaims").getAsJsonArray("xui").get(0).getAsJsonObject().get("uhs").getAsString()), new XblSisuTokens.SisuXstsToken(Instant.parse(obj.getAsJsonObject("AuthorizationToken").get("NotAfter").getAsString()).toEpochMilli(), obj.getAsJsonObject("AuthorizationToken").get("Token").getAsString(), obj.getAsJsonObject("AuthorizationToken").getAsJsonObject("DisplayClaims").getAsJsonArray("xui").get(0).getAsJsonObject().get("uhs").getAsString()), initialXblSession);
        MinecraftAuth.LOGGER.info("Got XBL Title+User+XSTS Token, expires: " + Instant.ofEpochMilli(xblSisuTokens.getExpireTimeMs()).atZone(ZoneId.systemDefault()));
        return xblSisuTokens;
    }

    @Override
    public StepXblSisuAuthentication.XblSisuTokens fromJson(final JsonObject json) {
        final StepInitialXblSession.InitialXblSession initialXblSession = this.prevStep != null ? this.prevStep.fromJson(json.getAsJsonObject(this.prevStep.name)) : null;
        return new StepXblSisuAuthentication.XblSisuTokens(XblSisuTokens.SisuTitleToken.fromJson(json.getAsJsonObject("titleToken")), XblSisuTokens.SisuUserToken.fromJson(json.getAsJsonObject("userToken")), XblSisuTokens.SisuXstsToken.fromJson(json.getAsJsonObject("xstsToken")), initialXblSession);
    }

    @Override
    public JsonObject toJson(final StepXblSisuAuthentication.XblSisuTokens xblSisuTokens) {
        final JsonObject json = new JsonObject();
        json.add("titleToken", XblSisuTokens.SisuTitleToken.toJson(xblSisuTokens.titleToken));
        json.add("userToken", XblSisuTokens.SisuUserToken.toJson(xblSisuTokens.userToken));
        json.add("xstsToken", XblSisuTokens.SisuXstsToken.toJson(xblSisuTokens.xstsToken));
        if (this.prevStep != null) json.add(this.prevStep.name, this.prevStep.toJson(xblSisuTokens.initialXblSession));
        return json;
    }


    public static final class XblSisuTokens extends StepXblXstsToken.XblXsts<StepInitialXblSession.InitialXblSession> {
        private final SisuTitleToken titleToken;
        private final SisuUserToken userToken;
        private final SisuXstsToken xstsToken;
        private final StepInitialXblSession.InitialXblSession initialXblSession;

        @Override
        public long getExpireTimeMs() {
            return Math.min(Math.min(this.xstsToken.expireTimeMs, this.titleToken.expireTimeMs), this.userToken.expireTimeMs);
        }

        @Override
        public String getToken() {
            return this.xstsToken.token;
        }

        @Override
        public String getUserHash() {
            return this.xstsToken.userHash;
        }

        @Override
        public StepFullXblSession.FullXblSession getFullXblSession() {
            final StepXblUserToken.XblUserToken userToken = new StepXblUserToken.XblUserToken(this.userToken.expireTimeMs, this.userToken.token, this.userToken.userHash, this.initialXblSession);
            final StepXblTitleToken.XblTitleToken titleToken = new StepXblTitleToken.XblTitleToken(this.titleToken.expireTimeMs, this.titleToken.token, this.titleToken.titleId, this.initialXblSession);
            return new StepFullXblSession.FullXblSession(userToken, titleToken);
        }

        @Override
        protected StepInitialXblSession.InitialXblSession prevResult() {
            return this.initialXblSession;
        }

        @Override
        public boolean isExpired() {
            return this.getExpireTimeMs() <= System.currentTimeMillis();
        }


        public static final class SisuTitleToken {
            private final long expireTimeMs;
            private final String token;
            private final String titleId;

            public static SisuTitleToken fromJson(final JsonObject json) {
                return new SisuTitleToken(json.get("expireTimeMs").getAsLong(), json.get("token").getAsString(), json.get("titleId").getAsString());
            }

            public static JsonObject toJson(final SisuTitleToken sisuTitleToken) {
                final JsonObject json = new JsonObject();
                json.addProperty("expireTimeMs", sisuTitleToken.expireTimeMs);
                json.addProperty("token", sisuTitleToken.token);
                json.addProperty("titleId", sisuTitleToken.titleId);
                return json;
            }

            public SisuTitleToken(final long expireTimeMs, final String token, final String titleId) {
                this.expireTimeMs = expireTimeMs;
                this.token = token;
                this.titleId = titleId;
            }

            public long getExpireTimeMs() {
                return this.expireTimeMs;
            }

            public String getToken() {
                return this.token;
            }

            public String getTitleId() {
                return this.titleId;
            }

            @Override
            public boolean equals(final Object o) {
                if (o == this) return true;
                if (!(o instanceof StepXblSisuAuthentication.XblSisuTokens.SisuTitleToken)) return false;
                final StepXblSisuAuthentication.XblSisuTokens.SisuTitleToken other = (StepXblSisuAuthentication.XblSisuTokens.SisuTitleToken) o;
                if (this.getExpireTimeMs() != other.getExpireTimeMs()) return false;
                final Object this$token = this.getToken();
                final Object other$token = other.getToken();
                if (this$token == null ? other$token != null : !this$token.equals(other$token)) return false;
                final Object this$titleId = this.getTitleId();
                final Object other$titleId = other.getTitleId();
                if (this$titleId == null ? other$titleId != null : !this$titleId.equals(other$titleId)) return false;
                return true;
            }

            @Override
            public int hashCode() {
                final int PRIME = 59;
                int result = 1;
                final long $expireTimeMs = this.getExpireTimeMs();
                result = result * PRIME + (int) ($expireTimeMs >>> 32 ^ $expireTimeMs);
                final Object $token = this.getToken();
                result = result * PRIME + ($token == null ? 43 : $token.hashCode());
                final Object $titleId = this.getTitleId();
                result = result * PRIME + ($titleId == null ? 43 : $titleId.hashCode());
                return result;
            }

            @Override
            public String toString() {
                return "StepXblSisuAuthentication.XblSisuTokens.SisuTitleToken(expireTimeMs=" + this.getExpireTimeMs() + ", token=" + this.getToken() + ", titleId=" + this.getTitleId() + ")";
            }
        }


        public static final class SisuUserToken {
            private final long expireTimeMs;
            private final String token;
            private final String userHash;

            public static SisuUserToken fromJson(final JsonObject json) {
                return new SisuUserToken(json.get("expireTimeMs").getAsLong(), json.get("token").getAsString(), json.get("userHash").getAsString());
            }

            public static JsonObject toJson(final SisuUserToken sisuUserToken) {
                final JsonObject json = new JsonObject();
                json.addProperty("expireTimeMs", sisuUserToken.expireTimeMs);
                json.addProperty("token", sisuUserToken.token);
                json.addProperty("userHash", sisuUserToken.userHash);
                return json;
            }

            public SisuUserToken(final long expireTimeMs, final String token, final String userHash) {
                this.expireTimeMs = expireTimeMs;
                this.token = token;
                this.userHash = userHash;
            }

            public long getExpireTimeMs() {
                return this.expireTimeMs;
            }

            public String getToken() {
                return this.token;
            }

            public String getUserHash() {
                return this.userHash;
            }

            @Override
            public boolean equals(final Object o) {
                if (o == this) return true;
                if (!(o instanceof StepXblSisuAuthentication.XblSisuTokens.SisuUserToken)) return false;
                final StepXblSisuAuthentication.XblSisuTokens.SisuUserToken other = (StepXblSisuAuthentication.XblSisuTokens.SisuUserToken) o;
                if (this.getExpireTimeMs() != other.getExpireTimeMs()) return false;
                final Object this$token = this.getToken();
                final Object other$token = other.getToken();
                if (this$token == null ? other$token != null : !this$token.equals(other$token)) return false;
                final Object this$userHash = this.getUserHash();
                final Object other$userHash = other.getUserHash();
                if (this$userHash == null ? other$userHash != null : !this$userHash.equals(other$userHash)) return false;
                return true;
            }

            @Override
            public int hashCode() {
                final int PRIME = 59;
                int result = 1;
                final long $expireTimeMs = this.getExpireTimeMs();
                result = result * PRIME + (int) ($expireTimeMs >>> 32 ^ $expireTimeMs);
                final Object $token = this.getToken();
                result = result * PRIME + ($token == null ? 43 : $token.hashCode());
                final Object $userHash = this.getUserHash();
                result = result * PRIME + ($userHash == null ? 43 : $userHash.hashCode());
                return result;
            }

            @Override
            public String toString() {
                return "StepXblSisuAuthentication.XblSisuTokens.SisuUserToken(expireTimeMs=" + this.getExpireTimeMs() + ", token=" + this.getToken() + ", userHash=" + this.getUserHash() + ")";
            }
        }


        public static final class SisuXstsToken {
            private final long expireTimeMs;
            private final String token;
            private final String userHash;

            public static SisuXstsToken fromJson(final JsonObject json) {
                return new SisuXstsToken(json.get("expireTimeMs").getAsLong(), json.get("token").getAsString(), json.get("userHash").getAsString());
            }

            public static JsonObject toJson(final SisuXstsToken sisuXstsToken) {
                final JsonObject json = new JsonObject();
                json.addProperty("expireTimeMs", sisuXstsToken.expireTimeMs);
                json.addProperty("token", sisuXstsToken.token);
                json.addProperty("userHash", sisuXstsToken.userHash);
                return json;
            }

            public SisuXstsToken(final long expireTimeMs, final String token, final String userHash) {
                this.expireTimeMs = expireTimeMs;
                this.token = token;
                this.userHash = userHash;
            }

            public long getExpireTimeMs() {
                return this.expireTimeMs;
            }

            public String getToken() {
                return this.token;
            }

            public String getUserHash() {
                return this.userHash;
            }

            @Override
            public boolean equals(final Object o) {
                if (o == this) return true;
                if (!(o instanceof StepXblSisuAuthentication.XblSisuTokens.SisuXstsToken)) return false;
                final StepXblSisuAuthentication.XblSisuTokens.SisuXstsToken other = (StepXblSisuAuthentication.XblSisuTokens.SisuXstsToken) o;
                if (this.getExpireTimeMs() != other.getExpireTimeMs()) return false;
                final Object this$token = this.getToken();
                final Object other$token = other.getToken();
                if (this$token == null ? other$token != null : !this$token.equals(other$token)) return false;
                final Object this$userHash = this.getUserHash();
                final Object other$userHash = other.getUserHash();
                if (this$userHash == null ? other$userHash != null : !this$userHash.equals(other$userHash)) return false;
                return true;
            }

            @Override
            public int hashCode() {
                final int PRIME = 59;
                int result = 1;
                final long $expireTimeMs = this.getExpireTimeMs();
                result = result * PRIME + (int) ($expireTimeMs >>> 32 ^ $expireTimeMs);
                final Object $token = this.getToken();
                result = result * PRIME + ($token == null ? 43 : $token.hashCode());
                final Object $userHash = this.getUserHash();
                result = result * PRIME + ($userHash == null ? 43 : $userHash.hashCode());
                return result;
            }

            @Override
            public String toString() {
                return "StepXblSisuAuthentication.XblSisuTokens.SisuXstsToken(expireTimeMs=" + this.getExpireTimeMs() + ", token=" + this.getToken() + ", userHash=" + this.getUserHash() + ")";
            }
        }

        public XblSisuTokens(final SisuTitleToken titleToken, final SisuUserToken userToken, final SisuXstsToken xstsToken, final StepInitialXblSession.InitialXblSession initialXblSession) {
            this.titleToken = titleToken;
            this.userToken = userToken;
            this.xstsToken = xstsToken;
            this.initialXblSession = initialXblSession;
        }

        public SisuTitleToken getTitleToken() {
            return this.titleToken;
        }

        public SisuUserToken getUserToken() {
            return this.userToken;
        }

        public SisuXstsToken getXstsToken() {
            return this.xstsToken;
        }

        public StepInitialXblSession.InitialXblSession getInitialXblSession() {
            return this.initialXblSession;
        }

        @Override
        public String toString() {
            return "StepXblSisuAuthentication.XblSisuTokens(titleToken=" + this.getTitleToken() + ", userToken=" + this.getUserToken() + ", xstsToken=" + this.getXstsToken() + ", initialXblSession=" + this.getInitialXblSession() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepXblSisuAuthentication.XblSisuTokens)) return false;
            final StepXblSisuAuthentication.XblSisuTokens other = (StepXblSisuAuthentication.XblSisuTokens) o;
            if (!other.canEqual((Object) this)) return false;
            final Object this$titleToken = this.getTitleToken();
            final Object other$titleToken = other.getTitleToken();
            if (this$titleToken == null ? other$titleToken != null : !this$titleToken.equals(other$titleToken)) return false;
            final Object this$userToken = this.getUserToken();
            final Object other$userToken = other.getUserToken();
            if (this$userToken == null ? other$userToken != null : !this$userToken.equals(other$userToken)) return false;
            final Object this$xstsToken = this.getXstsToken();
            final Object other$xstsToken = other.getXstsToken();
            if (this$xstsToken == null ? other$xstsToken != null : !this$xstsToken.equals(other$xstsToken)) return false;
            final Object this$initialXblSession = this.getInitialXblSession();
            final Object other$initialXblSession = other.getInitialXblSession();
            if (this$initialXblSession == null ? other$initialXblSession != null : !this$initialXblSession.equals(other$initialXblSession)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepXblSisuAuthentication.XblSisuTokens;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final Object $titleToken = this.getTitleToken();
            result = result * PRIME + ($titleToken == null ? 43 : $titleToken.hashCode());
            final Object $userToken = this.getUserToken();
            result = result * PRIME + ($userToken == null ? 43 : $userToken.hashCode());
            final Object $xstsToken = this.getXstsToken();
            result = result * PRIME + ($xstsToken == null ? 43 : $xstsToken.hashCode());
            final Object $initialXblSession = this.getInitialXblSession();
            result = result * PRIME + ($initialXblSession == null ? 43 : $initialXblSession.hashCode());
            return result;
        }
    }
}
