/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.xbl;

import com.google.gson.JsonObject;
import net.lenni0451.commons.httpclient.HttpClient;
import net.lenni0451.commons.httpclient.requests.impl.PostRequest;
import net.raphimc.minecraftauth.MinecraftAuth;
import net.raphimc.minecraftauth.responsehandler.XblResponseHandler;
import net.raphimc.minecraftauth.step.AbstractStep;
import net.raphimc.minecraftauth.step.xbl.session.StepInitialXblSession;
import net.raphimc.minecraftauth.util.CryptUtil;
import net.raphimc.minecraftauth.util.JsonContent;
import java.time.Instant;
import java.time.ZoneId;

public class StepXblTitleToken extends AbstractStep<StepInitialXblSession.InitialXblSession, StepXblTitleToken.XblTitleToken> {
    public static final String XBL_TITLE_URL = "https://title.auth.xboxlive.com/title/authenticate";

    public StepXblTitleToken(final AbstractStep<?, StepInitialXblSession.InitialXblSession> prevStep) {
        super("titleToken", prevStep);
    }

    @Override
    public XblTitleToken applyStep(final HttpClient httpClient, final StepInitialXblSession.InitialXblSession initialXblSession) throws Exception {
        MinecraftAuth.LOGGER.info("Authenticating title with Xbox Live...");
        if (initialXblSession.getXblDeviceToken() == null) {
            throw new IllegalStateException("An XBL Device Token is needed for Title authentication");
        }
        if (!initialXblSession.getMsaToken().getMsaCode().getApplicationDetails().isTitleClientId()) {
            throw new IllegalStateException("A Title Client ID is needed for Title authentication");
        }
        final JsonObject postData = new JsonObject();
        final JsonObject properties = new JsonObject();
        properties.addProperty("AuthMethod", "RPS");
        properties.addProperty("SiteName", "user.auth.xboxlive.com");
        properties.addProperty("RpsTicket", "t=" + initialXblSession.getMsaToken().getAccessToken());
        properties.addProperty("DeviceToken", initialXblSession.getXblDeviceToken().getToken());
        properties.add("ProofKey", CryptUtil.getProofKey(initialXblSession.getXblDeviceToken().getPublicKey()));
        postData.add("Properties", properties);
        postData.addProperty("RelyingParty", "http://auth.xboxlive.com");
        postData.addProperty("TokenType", "JWT");
        final PostRequest postRequest = new PostRequest(XBL_TITLE_URL);
        postRequest.setContent(new JsonContent(postData));
        postRequest.setHeader("x-xbl-contract-version", "1");
        postRequest.setHeader(CryptUtil.getSignatureHeader(postRequest, initialXblSession.getXblDeviceToken().getPrivateKey()));
        final JsonObject obj = httpClient.execute(postRequest, new XblResponseHandler());
        final XblTitleToken xblTitleToken = new XblTitleToken(Instant.parse(obj.get("NotAfter").getAsString()).toEpochMilli(), obj.get("Token").getAsString(), obj.getAsJsonObject("DisplayClaims").getAsJsonObject("xti").get("tid").getAsString(), initialXblSession);
        MinecraftAuth.LOGGER.info("Got XBL Title Token, expires: " + Instant.ofEpochMilli(xblTitleToken.getExpireTimeMs()).atZone(ZoneId.systemDefault()));
        return xblTitleToken;
    }

    @Override
    public XblTitleToken fromJson(final JsonObject json) {
        final StepInitialXblSession.InitialXblSession initialXblSession = this.prevStep != null ? this.prevStep.fromJson(json.getAsJsonObject(this.prevStep.name)) : null;
        return new XblTitleToken(json.get("expireTimeMs").getAsLong(), json.get("token").getAsString(), json.get("titleId").getAsString(), initialXblSession);
    }

    @Override
    public JsonObject toJson(final XblTitleToken xblTitleToken) {
        final JsonObject json = new JsonObject();
        json.addProperty("expireTimeMs", xblTitleToken.expireTimeMs);
        json.addProperty("token", xblTitleToken.token);
        json.addProperty("titleId", xblTitleToken.titleId);
        if (this.prevStep != null) json.add(this.prevStep.name, this.prevStep.toJson(xblTitleToken.initialXblSession));
        return json;
    }


    public static final class XblTitleToken extends AbstractStep.StepResult<StepInitialXblSession.InitialXblSession> {
        private final long expireTimeMs;
        private final String token;
        private final String titleId;
        private final StepInitialXblSession.InitialXblSession initialXblSession;

        @Override
        protected StepInitialXblSession.InitialXblSession prevResult() {
            return this.initialXblSession;
        }

        @Override
        public boolean isExpired() {
            return this.expireTimeMs <= System.currentTimeMillis();
        }

        public XblTitleToken(final long expireTimeMs, final String token, final String titleId, final StepInitialXblSession.InitialXblSession initialXblSession) {
            this.expireTimeMs = expireTimeMs;
            this.token = token;
            this.titleId = titleId;
            this.initialXblSession = initialXblSession;
        }

        public long getExpireTimeMs() {
            return this.expireTimeMs;
        }

        public String getToken() {
            return this.token;
        }

        public String getTitleId() {
            return this.titleId;
        }

        public StepInitialXblSession.InitialXblSession getInitialXblSession() {
            return this.initialXblSession;
        }

        @Override
        public String toString() {
            return "StepXblTitleToken.XblTitleToken(expireTimeMs=" + this.getExpireTimeMs() + ", token=" + this.getToken() + ", titleId=" + this.getTitleId() + ", initialXblSession=" + this.getInitialXblSession() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepXblTitleToken.XblTitleToken)) return false;
            final StepXblTitleToken.XblTitleToken other = (StepXblTitleToken.XblTitleToken) o;
            if (!other.canEqual((Object) this)) return false;
            if (this.getExpireTimeMs() != other.getExpireTimeMs()) return false;
            final Object this$token = this.getToken();
            final Object other$token = other.getToken();
            if (this$token == null ? other$token != null : !this$token.equals(other$token)) return false;
            final Object this$titleId = this.getTitleId();
            final Object other$titleId = other.getTitleId();
            if (this$titleId == null ? other$titleId != null : !this$titleId.equals(other$titleId)) return false;
            final Object this$initialXblSession = this.getInitialXblSession();
            final Object other$initialXblSession = other.getInitialXblSession();
            if (this$initialXblSession == null ? other$initialXblSession != null : !this$initialXblSession.equals(other$initialXblSession)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepXblTitleToken.XblTitleToken;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final long $expireTimeMs = this.getExpireTimeMs();
            result = result * PRIME + (int) ($expireTimeMs >>> 32 ^ $expireTimeMs);
            final Object $token = this.getToken();
            result = result * PRIME + ($token == null ? 43 : $token.hashCode());
            final Object $titleId = this.getTitleId();
            result = result * PRIME + ($titleId == null ? 43 : $titleId.hashCode());
            final Object $initialXblSession = this.getInitialXblSession();
            result = result * PRIME + ($initialXblSession == null ? 43 : $initialXblSession.hashCode());
            return result;
        }
    }
}
