/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.xbl;

import com.google.gson.JsonObject;
import net.lenni0451.commons.httpclient.HttpClient;
import net.lenni0451.commons.httpclient.requests.impl.PostRequest;
import net.raphimc.minecraftauth.MinecraftAuth;
import net.raphimc.minecraftauth.responsehandler.XblResponseHandler;
import net.raphimc.minecraftauth.step.AbstractStep;
import net.raphimc.minecraftauth.step.xbl.session.StepInitialXblSession;
import net.raphimc.minecraftauth.util.CryptUtil;
import net.raphimc.minecraftauth.util.JsonContent;
import java.time.Instant;
import java.time.ZoneId;

public class StepXblUserToken extends AbstractStep<StepInitialXblSession.InitialXblSession, StepXblUserToken.XblUserToken> {
    public static final String XBL_USER_URL = "https://user.auth.xboxlive.com/user/authenticate";

    public StepXblUserToken(final AbstractStep<?, StepInitialXblSession.InitialXblSession> prevStep) {
        super("xblUserToken", prevStep);
    }

    @Override
    public XblUserToken applyStep(final HttpClient httpClient, final StepInitialXblSession.InitialXblSession initialXblSession) throws Exception {
        MinecraftAuth.LOGGER.info("Authenticating user with Xbox Live...");
        final JsonObject postData = new JsonObject();
        final JsonObject properties = new JsonObject();
        properties.addProperty("AuthMethod", "RPS");
        properties.addProperty("SiteName", "user.auth.xboxlive.com");
        properties.addProperty("RpsTicket", (initialXblSession.getMsaToken().getMsaCode().getApplicationDetails().isTitleClientId() ? "t=" : "d=") + initialXblSession.getMsaToken().getAccessToken());
        if (initialXblSession.getXblDeviceToken() != null) {
            properties.add("ProofKey", CryptUtil.getProofKey(initialXblSession.getXblDeviceToken().getPublicKey()));
        }
        postData.add("Properties", properties);
        postData.addProperty("RelyingParty", "http://auth.xboxlive.com");
        postData.addProperty("TokenType", "JWT");
        final PostRequest postRequest = new PostRequest(XBL_USER_URL);
        postRequest.setContent(new JsonContent(postData));
        postRequest.setHeader("x-xbl-contract-version", "1");
        if (initialXblSession.getXblDeviceToken() != null) {
            postRequest.setHeader(CryptUtil.getSignatureHeader(postRequest, initialXblSession.getXblDeviceToken().getPrivateKey()));
        }
        final JsonObject obj = httpClient.execute(postRequest, new XblResponseHandler());
        final XblUserToken xblUserToken = new XblUserToken(Instant.parse(obj.get("NotAfter").getAsString()).toEpochMilli(), obj.get("Token").getAsString(), obj.getAsJsonObject("DisplayClaims").getAsJsonArray("xui").get(0).getAsJsonObject().get("uhs").getAsString(), initialXblSession);
        MinecraftAuth.LOGGER.info("Got XBL User Token, expires: " + Instant.ofEpochMilli(xblUserToken.getExpireTimeMs()).atZone(ZoneId.systemDefault()));
        return xblUserToken;
    }

    @Override
    public XblUserToken fromJson(final JsonObject json) {
        final StepInitialXblSession.InitialXblSession initialXblSession = this.prevStep != null ? this.prevStep.fromJson(json.getAsJsonObject(this.prevStep.name)) : null;
        return new XblUserToken(json.get("expireTimeMs").getAsLong(), json.get("token").getAsString(), json.get("userHash").getAsString(), initialXblSession);
    }

    @Override
    public JsonObject toJson(final XblUserToken xblUserToken) {
        final JsonObject json = new JsonObject();
        json.addProperty("expireTimeMs", xblUserToken.expireTimeMs);
        json.addProperty("token", xblUserToken.token);
        json.addProperty("userHash", xblUserToken.userHash);
        if (this.prevStep != null) json.add(this.prevStep.name, this.prevStep.toJson(xblUserToken.initialXblSession));
        return json;
    }


    public static final class XblUserToken extends AbstractStep.StepResult<StepInitialXblSession.InitialXblSession> {
        private final long expireTimeMs;
        private final String token;
        private final String userHash;
        private final StepInitialXblSession.InitialXblSession initialXblSession;

        @Override
        protected StepInitialXblSession.InitialXblSession prevResult() {
            return this.initialXblSession;
        }

        @Override
        public boolean isExpired() {
            return this.expireTimeMs <= System.currentTimeMillis();
        }

        public XblUserToken(final long expireTimeMs, final String token, final String userHash, final StepInitialXblSession.InitialXblSession initialXblSession) {
            this.expireTimeMs = expireTimeMs;
            this.token = token;
            this.userHash = userHash;
            this.initialXblSession = initialXblSession;
        }

        public long getExpireTimeMs() {
            return this.expireTimeMs;
        }

        public String getToken() {
            return this.token;
        }

        public String getUserHash() {
            return this.userHash;
        }

        public StepInitialXblSession.InitialXblSession getInitialXblSession() {
            return this.initialXblSession;
        }

        @Override
        public String toString() {
            return "StepXblUserToken.XblUserToken(expireTimeMs=" + this.getExpireTimeMs() + ", token=" + this.getToken() + ", userHash=" + this.getUserHash() + ", initialXblSession=" + this.getInitialXblSession() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepXblUserToken.XblUserToken)) return false;
            final StepXblUserToken.XblUserToken other = (StepXblUserToken.XblUserToken) o;
            if (!other.canEqual((Object) this)) return false;
            if (this.getExpireTimeMs() != other.getExpireTimeMs()) return false;
            final Object this$token = this.getToken();
            final Object other$token = other.getToken();
            if (this$token == null ? other$token != null : !this$token.equals(other$token)) return false;
            final Object this$userHash = this.getUserHash();
            final Object other$userHash = other.getUserHash();
            if (this$userHash == null ? other$userHash != null : !this$userHash.equals(other$userHash)) return false;
            final Object this$initialXblSession = this.getInitialXblSession();
            final Object other$initialXblSession = other.getInitialXblSession();
            if (this$initialXblSession == null ? other$initialXblSession != null : !this$initialXblSession.equals(other$initialXblSession)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepXblUserToken.XblUserToken;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final long $expireTimeMs = this.getExpireTimeMs();
            result = result * PRIME + (int) ($expireTimeMs >>> 32 ^ $expireTimeMs);
            final Object $token = this.getToken();
            result = result * PRIME + ($token == null ? 43 : $token.hashCode());
            final Object $userHash = this.getUserHash();
            result = result * PRIME + ($userHash == null ? 43 : $userHash.hashCode());
            final Object $initialXblSession = this.getInitialXblSession();
            result = result * PRIME + ($initialXblSession == null ? 43 : $initialXblSession.hashCode());
            return result;
        }
    }
}
