/*
 * This file is part of MinecraftAuth - https://github.com/RaphiMC/MinecraftAuth
 * Copyright (C) 2022-2024 RK_01/RaphiMC and contributors
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package net.raphimc.minecraftauth.step.xbl;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import net.lenni0451.commons.httpclient.HttpClient;
import net.lenni0451.commons.httpclient.requests.impl.PostRequest;
import net.raphimc.minecraftauth.MinecraftAuth;
import net.raphimc.minecraftauth.responsehandler.XblResponseHandler;
import net.raphimc.minecraftauth.step.AbstractStep;
import net.raphimc.minecraftauth.step.xbl.session.StepFullXblSession;
import net.raphimc.minecraftauth.step.xbl.session.StepInitialXblSession;
import net.raphimc.minecraftauth.util.CryptUtil;
import net.raphimc.minecraftauth.util.JsonContent;
import java.time.Instant;
import java.time.ZoneId;

public class StepXblXstsToken extends AbstractStep<StepFullXblSession.FullXblSession, StepXblXstsToken.XblXstsToken> {
    public static final String XBL_XSTS_URL = "https://xsts.auth.xboxlive.com/xsts/authorize";
    private final String relyingParty;

    public StepXblXstsToken(final AbstractStep<?, StepFullXblSession.FullXblSession> prevStep, final String relyingParty) {
        this("xblXstsToken", prevStep, relyingParty);
    }

    public StepXblXstsToken(final String name, final AbstractStep<?, StepFullXblSession.FullXblSession> prevStep, final String relyingParty) {
        super(name, prevStep);
        this.relyingParty = relyingParty;
    }

    @Override
    public XblXstsToken applyStep(final HttpClient httpClient, final StepFullXblSession.FullXblSession fullXblSession) throws Exception {
        MinecraftAuth.LOGGER.info("Requesting XSTS Token...");
        final JsonObject postData = new JsonObject();
        final JsonObject properties = new JsonObject();
        properties.addProperty("SandboxId", "RETAIL");
        final JsonArray userTokens = new JsonArray();
        userTokens.add(new JsonPrimitive(fullXblSession.getXblUserToken().getToken()));
        properties.add("UserTokens", userTokens);
        if (fullXblSession.getXblTitleToken() != null) {
            properties.addProperty("TitleToken", fullXblSession.getXblTitleToken().getToken());
            properties.addProperty("DeviceToken", fullXblSession.getXblTitleToken().getInitialXblSession().getXblDeviceToken().getToken());
        }
        postData.add("Properties", properties);
        postData.addProperty("RelyingParty", this.relyingParty);
        postData.addProperty("TokenType", "JWT");
        final PostRequest postRequest = new PostRequest(XBL_XSTS_URL);
        postRequest.setContent(new JsonContent(postData));
        postRequest.setHeader("x-xbl-contract-version", "1");
        if (fullXblSession.getXblTitleToken() != null) {
            postRequest.setHeader(CryptUtil.getSignatureHeader(postRequest, fullXblSession.getXblTitleToken().getInitialXblSession().getXblDeviceToken().getPrivateKey()));
        }
        final JsonObject obj = httpClient.execute(postRequest, new XblResponseHandler());
        final XblXstsToken xblXstsToken = new XblXstsToken(Instant.parse(obj.get("NotAfter").getAsString()).toEpochMilli(), obj.get("Token").getAsString(), obj.getAsJsonObject("DisplayClaims").getAsJsonArray("xui").get(0).getAsJsonObject().get("uhs").getAsString(), fullXblSession);
        MinecraftAuth.LOGGER.info("Got XSTS Token, expires: " + Instant.ofEpochMilli(xblXstsToken.getExpireTimeMs()).atZone(ZoneId.systemDefault()));
        return xblXstsToken;
    }

    @Override
    public XblXstsToken fromJson(final JsonObject json) {
        final StepFullXblSession.FullXblSession fullXblSession = this.prevStep != null ? this.prevStep.fromJson(json.getAsJsonObject(this.prevStep.name)) : null;
        return new XblXstsToken(json.get("expireTimeMs").getAsLong(), json.get("token").getAsString(), json.get("userHash").getAsString(), fullXblSession);
    }

    @Override
    public JsonObject toJson(final StepXblXstsToken.XblXstsToken xblXstsToken) {
        final JsonObject json = new JsonObject();
        json.addProperty("expireTimeMs", xblXstsToken.expireTimeMs);
        json.addProperty("token", xblXstsToken.token);
        json.addProperty("userHash", xblXstsToken.userHash);
        if (this.prevStep != null) json.add(this.prevStep.name, this.prevStep.toJson(xblXstsToken.fullXblSession));
        return json;
    }


    public static final class XblXstsToken extends XblXsts<StepFullXblSession.FullXblSession> {
        private final long expireTimeMs;
        private final String token;
        private final String userHash;
        private final StepFullXblSession.FullXblSession fullXblSession;

        @Override
        protected StepFullXblSession.FullXblSession prevResult() {
            return this.fullXblSession;
        }

        @Override
        public boolean isExpired() {
            return this.expireTimeMs <= System.currentTimeMillis();
        }

        public XblXstsToken(final long expireTimeMs, final String token, final String userHash, final StepFullXblSession.FullXblSession fullXblSession) {
            this.expireTimeMs = expireTimeMs;
            this.token = token;
            this.userHash = userHash;
            this.fullXblSession = fullXblSession;
        }

        public long getExpireTimeMs() {
            return this.expireTimeMs;
        }

        public String getToken() {
            return this.token;
        }

        public String getUserHash() {
            return this.userHash;
        }

        public StepFullXblSession.FullXblSession getFullXblSession() {
            return this.fullXblSession;
        }

        @Override
        public String toString() {
            return "StepXblXstsToken.XblXstsToken(expireTimeMs=" + this.getExpireTimeMs() + ", token=" + this.getToken() + ", userHash=" + this.getUserHash() + ", fullXblSession=" + this.getFullXblSession() + ")";
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof StepXblXstsToken.XblXstsToken)) return false;
            final StepXblXstsToken.XblXstsToken other = (StepXblXstsToken.XblXstsToken) o;
            if (!other.canEqual((Object) this)) return false;
            if (this.getExpireTimeMs() != other.getExpireTimeMs()) return false;
            final Object this$token = this.getToken();
            final Object other$token = other.getToken();
            if (this$token == null ? other$token != null : !this$token.equals(other$token)) return false;
            final Object this$userHash = this.getUserHash();
            final Object other$userHash = other.getUserHash();
            if (this$userHash == null ? other$userHash != null : !this$userHash.equals(other$userHash)) return false;
            final Object this$fullXblSession = this.getFullXblSession();
            final Object other$fullXblSession = other.getFullXblSession();
            if (this$fullXblSession == null ? other$fullXblSession != null : !this$fullXblSession.equals(other$fullXblSession)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof StepXblXstsToken.XblXstsToken;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final long $expireTimeMs = this.getExpireTimeMs();
            result = result * PRIME + (int) ($expireTimeMs >>> 32 ^ $expireTimeMs);
            final Object $token = this.getToken();
            result = result * PRIME + ($token == null ? 43 : $token.hashCode());
            final Object $userHash = this.getUserHash();
            result = result * PRIME + ($userHash == null ? 43 : $userHash.hashCode());
            final Object $fullXblSession = this.getFullXblSession();
            result = result * PRIME + ($fullXblSession == null ? 43 : $fullXblSession.hashCode());
            return result;
        }
    }


    public static abstract class XblXsts<P extends AbstractStep.StepResult<?>> extends AbstractStep.StepResult<P> {
        public abstract long getExpireTimeMs();

        public abstract String getToken();

        public abstract String getUserHash();

        public String getServiceToken() {
            return this.getUserHash() + ';' + this.getToken();
        }

        public abstract StepFullXblSession.FullXblSession getFullXblSession();

        public StepInitialXblSession.InitialXblSession getInitialXblSession() {
            return this.getFullXblSession().getXblUserToken().getInitialXblSession();
        }
    }
}
