package tech.beepbeep.beep_loader

import bolts.Task
import tech.beepbeep.beep_commons.CoreService
import tech.beepbeep.beep_commons.MachineControllerService
import tech.beepbeep.beep_commons.MachineState
import tech.beepbeep.beep_commons.PaymentService
import tech.beepbeep.beep_commons.payment.PaymentResultListener
import tech.beepbeep.beep_commons.serial.bean.SerialPort
import tech.beepbeep.beep_commons.util.*
import tech.beepbeep.beep_commons.util.Utils.Companion.printLog
import java.io.File
import java.io.PrintWriter
import java.io.StringWriter
import java.util.*
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors
import java.util.concurrent.ThreadFactory

/**
 * This class will handle the logic of loading all jar files.
 *
 * command for build jar:
 *
 * step 1: build jar with utils.jar
 *
 * ant -f C:\Users\yunze\Desktop\Loader\buildcore.xml
 * ant -f C:\Users\yunze\Desktop\Loader\buildpayment.xml
 * ant -f C:\Users\yunze\Desktop\Loader\buildmachine.xml
 *
 * step 2: dex jar for Android Dalvik VM
 *
 * dx --dex --output C:\Users\yunze\Desktop\Loader\core_dex.jar C:\Users\yunze\Desktop\Loader\final_core.jar
 * dx --dex --output C:\Users\yunze\Desktop\Loader\beep_core_service_dev_1.0.0.jar C:\Users\yunze\Desktop\Loader\service-beep-core-1.0.0-dev.jar
 * dx --dex --output C:\Users\yunze\Desktop\Loader\machine_dex.jar C:\Users\yunze\Desktop\Loader\final_machine.jar
 *
 * This Class support android only. testing
 */
class BeepLoaderManager {

//    private var myCounter = 0
    private lateinit var beepConnector: BeepConnector
    private lateinit var token: String
    private lateinit var distributor: String
    private lateinit var uuid: String
    var isDebug: Boolean = false
    private lateinit var serialPortList: List<SerialPort>
    private var threadPool: ExecutorService? = null
    private var registeredMessageProcessor =  ConcurrentHashMap<String, MessageProcessor>()
    @Volatile
    private var clientMessageProcessor: MessageProcessor? = null
    private var paymentListener: PaymentResultListener? = null
    private var stateCodeList = ArrayList<Pair<String, String>>()
    private var machineState: MachineState? = null
    private var description: String = ""

    private var coreFileLocation: String = ""

    private val collector = object: MessageProcessorCollector {
        override fun addMessageProcessor(tag: String, messageProcessor: MessageProcessor) {
            threadPool!!.execute {
                registeredMessageProcessor[tag] = messageProcessor
                coreService?.let {
                    for (service in it) {
                        service.setMessageProcessor(registeredMessageProcessor)
                    }
                }
                machineControllerService?.let {
                    for (service in it) {
                        service.setMessageProcessor(registeredMessageProcessor)
                    }
                }
                paymentService?.let {
                    for (service in it) {
                        service.setMessageProcessor(registeredMessageProcessor)
                    }
                }
            }
        }
    }

    /**
     * Listen jar file download process, if any new jar file download finished, load it immediately.
     * BeepLoader will run in app's thread by default and messageProcessor could be called from other services' thread,
     * we make all code running in threadPool instead of calling thread.
     *
     */
    private val messageProcessor = object : MessageProcessor {
        override fun processMessage(messageBody: MessageBody) {
            this@BeepLoaderManager.threadPool!!.execute {
                when (messageBody.what) {
                    Constants.Core.JAR_DOWNLOADING.value -> {
                        printLog("$TAG, Downloading jar: ${messageBody.obj1 as? String}")
                    }
                    Constants.Core.APPLICATION_DOWNLOAD_DONE.value -> {
                        printLog("$TAG, Got Application Download Done")
                        val applicationInfo = messageBody.obj1 as Pair<String, String>
                        //Store the intended application version here. This differs from Constants.APPLICATION
                        //in that the latter is set by the application itself to indicate what version it is.
                        beepConnector.secureStore(INTENDED_APP_PATH, applicationInfo.first)
                        beepConnector.secureStore(INTENDED_APP_VER, applicationInfo.second)

                        val coreJar = messageBody.obj2 as String?
                        if (coreJar == null){
                            printLog("$TAG, No additional Core specified, passing through")
                        } else {
                            printLog("$TAG, Additional Core specified ($coreJar). Setting and saving first")
                            val beepPath = beepConnector.secureRetrieve(Constants.BEEP_PATH_TAG) as? String
                            val jarPath = "$beepPath${File.separator}${messageBody.obj2 as? String}"

                            val currentJarPath = beepConnector.secureRetrieve(Constants.CORE_SERVICE) as? String

                            if (!currentJarPath.isNullOrEmpty() && currentJarPath != jarPath) {
                                File(currentJarPath).delete()
                            }

                            beepConnector.secureStore(Constants.CORE_SERVICE, jarPath)
                            beepConnector.secureStore(Constants.CORE_VERSION, messageBody.obj2 as String)
                        }
                    }
                    Constants.Core.PAYMENT_JAR_DOWNLOAD_DONE.value -> {
                        val beepPath = beepConnector.secureRetrieve(Constants.BEEP_PATH_TAG) as? String
                        val jarPath = "$beepPath${File.separator}${messageBody.obj1 as? String}"

                        val currentJarPath = beepConnector.secureRetrieve(Constants.PAYMENT_SERVICE) as? String
                        if (!currentJarPath.isNullOrEmpty() && currentJarPath != jarPath) {
                            File(currentJarPath).delete()
                        }

                        beepConnector.secureStore(Constants.PAYMENT_SERVICE,jarPath)
                        beepConnector.secureStore(Constants.PAYMENT_VERSION,messageBody.obj1 as String)

                        //todo: testing only
                        //jarPath = "$beepPath${File.separator}payment_dex.jar"

                        printLog("$TAG, Find PaymentService $jarPath and try to load it now")
                        paymentService?.reload()
                        paymentService = loadPaymentJar(jarPath)

                        paymentService?.let {
                            for (service in it) {
                                service.addPaymentListener(paymentListener)
                                machineState?.let {
                                    service.updateMachineState(it,description,*(stateCodeList.toTypedArray()))
                                }
                                service.setMessageProcessor(registeredMessageProcessor)
                                service.initPaymentService(collector,beepConnector,isDebug,threadPool)
                            }
                        }
                    }
                    Constants.Core.CORE_JAR_DOWNLOAD_DONE.value -> {
                        val beepPath = beepConnector.secureRetrieve(Constants.BEEP_PATH_TAG) as? String
                        val jarPath = "$beepPath${File.separator}${messageBody.obj1 as? String}"

                        val currentJarPath = beepConnector.secureRetrieve(Constants.CORE_SERVICE) as? String
                        if (!currentJarPath.isNullOrEmpty() && currentJarPath != jarPath) {
                            File(currentJarPath).delete()
                        }

                        beepConnector.secureStore(Constants.CORE_SERVICE, jarPath)
                        beepConnector.secureStore(Constants.CORE_VERSION, messageBody.obj1 as String)


                        //todo: testing only
                        //jarPath = "$beepPath${File.separator}core_dex.jar"

                        printLog("$TAG, Find CoreService $jarPath and try to load it now")
                        coreService?.reload()
                        coreService = loadCoreJar(jarPath)

                        coreService?.let {
                            for (service in coreService!!) {
                                service.setMessageProcessor(registeredMessageProcessor)
                                service.register(uuid,distributor,token,isDebug,beepConnector,serialPortList,threadPool,collector)
                                return@execute
                            }
                        } ?: run {
                            printLog("$TAG, Try To Load Core But Is Null")
                        }
                        coreFileLocation = ""
                    }
                    Constants.Core.MACHINE_CONTROLLER_JAR_DOWNLOAD_DONE.value -> {
                        val beepPath = beepConnector.secureRetrieve(Constants.BEEP_PATH_TAG) as? String
                        val jarPath = "$beepPath${File.separator}${messageBody.obj1 as? String}"

                        val currentJarPath = beepConnector.secureRetrieve(Constants.MACHINE_CONTROLLER_SERVICE) as? String
                        if (!currentJarPath.isNullOrEmpty() && currentJarPath != jarPath) {
                            File(currentJarPath).delete()
                        }

                        beepConnector.secureStore(Constants.MACHINE_CONTROLLER_SERVICE,jarPath)
                        beepConnector.secureStore(Constants.MACHINE_CONTROLLER_VERSION,messageBody.obj1 as String)

                        //todo: testing only
                        //jarPath = "$beepPath${File.separator}machine_dex.jar"

                        printLog("$TAG, Find MachineController $jarPath and try to load it now")
                        machineControllerService?.reload()
                        machineControllerService = loadMachineControllerJar(jarPath)

                        machineControllerService?.let {
                            for (service in machineControllerService!!) {
                                service.setMessageProcessor(registeredMessageProcessor)
                                service.initMachineControllerService(collector,beepConnector,isDebug, threadPool)
                            }
                        }
                    }
                    Constants.Core.FORCE_RESTART.value -> {
                        //For Force Restart, we reload the services so we need to send it here.
                        clientMessageProcessor?.let {
                            printLog("$TAG, Sending Message To Client")
                            it.processMessage(messageBody)
                        }

                        resetServices()
                        Task.delay(1000L).continueWith {
                            checkCoreJar()
                        }
                        return@execute
                    }
                    Constants.Core.DEVICE_CONFIGURATION_ERROR.value -> {
                        val errorMsg = messageBody.obj1 as String
                        printLog("$TAG: $errorMsg")
                    }
                    else -> {
                    }
                }
                clientMessageProcessor?.let {
                    printLog("$TAG, Sending Message To Client")
                    it.processMessage(messageBody)
                }
            }
        }
    }

    private val factoryExceptionHandler = Thread.UncaughtExceptionHandler { t, e ->
        printLog("$TAG, From Thread Pool: $e")
        if (e is ServiceConfigurationError && coreFileLocation.isNotEmpty()) {
            printLog("$TAG, Is Service Configuration Error, retrying")
            threadPool?.execute {
                printLog("$TAG: Waiting for 10 seconds before continuing")
                try {
                    Thread.sleep(10000)
                } catch (ignored: Exception){}
                File(coreFileLocation).delete()
                resetServices()
                checkCoreJar()
            }
        } else {
            Utils.recordCrash(e, TAG, Thread.currentThread().name)
        }
    }

    /**
     * Load all service
     *
     * @param beepConnector      callback
     * @param token              provided by beep
     * @param distributor        provided by beep (entity)
     * @param uuid               provided by manufacturer
     * @param isDebug
     * @param serialPortLists    provided by implementing app to indicate device serialPort info
     * @param threadPool         provided by implementing app
     */
    fun loadAllService(beepConnector: BeepConnector, token: String,
                       distributor: String, uuid: String, isDebug: Boolean,
                       serialPortLists: List<SerialPort>, threadPool: ExecutorService?,
                       clientMessageProcessor: MessageProcessor?) {
        printLog("$TAG, Try to load all Service")
        beepConnector.secureStore(Constants.BEEP_PATH_TAG, Constants.BEEP_PATH)
        this.beepConnector = beepConnector
        this.token = token
        this.distributor = distributor
        this.uuid = uuid
        this.isDebug = isDebug
        Utils.isDebug = isDebug
        this.serialPortList = serialPortLists
        this.threadPool = threadPool
        val factory: ThreadFactory = ThreadFactory { r ->
            val thread = Thread(r)
            thread.uncaughtExceptionHandler = factoryExceptionHandler
            thread
        }
        if (this.threadPool == null) {
            this.threadPool = Executors.newFixedThreadPool(10, factory)
        }

        registeredMessageProcessor[Constants.JAR_LOADER] = this.messageProcessor
        this.clientMessageProcessor = clientMessageProcessor

        val intendedApplicationVersion = beepConnector.secureRetrieve(INTENDED_APP_VER) as? String
        if (intendedApplicationVersion != null){
            val appCurrentVersion = beepConnector.secureRetrieve(Constants.APPLICATION) as? String
            if (appCurrentVersion != intendedApplicationVersion){
                printLog("$TAG, Current Application Version ($appCurrentVersion) does not " +
                        "match Intended Application Version ($intendedApplicationVersion)!")
                val intendedApplicationPath = beepConnector.secureRetrieve(INTENDED_APP_PATH) as String
                clientMessageProcessor?.let {
                    printLog("$TAG, Sending Message To Client")
                    //Force client to update application again. This is crucial because you need to ensure
                    //the right application is in place for the Core service being used.
                    it.processMessage(MessageBody(
                        what = Constants.Core.APPLICATION_DOWNLOAD_DONE.value,
                        obj1 = Pair(intendedApplicationPath, intendedApplicationVersion)
                    ))
                }
                return
            }
        }

        checkCoreJar()
    }

    private fun checkCoreJar(){
        printLog("$TAG: Checking Core Jar")
        val corePath = beepConnector.secureRetrieve(Constants.CORE_SERVICE) as? String

        if (corePath.isNullOrEmpty()) {
            this.threadPool!!.execute {
                val coreDownloader =
                    CoreDownloader()
                coreDownloader.getDownloadLink(Constants.BEEP_PATH, messageProcessor)
            }
        } else {
            val file = File(corePath)
            if (!file.exists()) {
                this.threadPool!!.execute {
                    val coreDownloader =
                        CoreDownloader()
                    coreDownloader.getDownloadLink(Constants.BEEP_PATH, messageProcessor)
                }
            } else {
                coreService?.let {
                    for (service in it) {
                        service.reset()
                    }
                }
                coreService?.reload()
                coreService = loadCoreJar(corePath)
                coreService?.let {
                    for (service in it) {
                        service.setMessageProcessor(registeredMessageProcessor)
                        service.register(uuid,distributor,token,isDebug,beepConnector,serialPortList,threadPool,collector)
                    }
                }
                coreFileLocation = ""
            }
        }
    }

    private fun resetServices(){
        printLog("$TAG: Resetting Services")

        try {
            coreService?.let {
                for (service in it) {
                    service.reset()
                }
            } ?: run {
                printLog("$TAG, Try To Reset Core Service But Is Null")
            }
        } catch (ex: Exception){
            val sw = StringWriter()
            val pw = PrintWriter(sw)
            ex.printStackTrace(pw)
            printLog("$TAG: Exception Resetting Core Service\n$sw")
        }

        try {
            paymentService?.let {
                for (service in it) {
                    service.reset()
                }
            } ?: run {
                printLog("$TAG, Try To Reset Payment Service But Is Null")
            }
        } catch (ex: Exception){
            val sw = StringWriter()
            val pw = PrintWriter(sw)
            ex.printStackTrace(pw)
            printLog("$TAG: Exception Resetting Payment Service\n$sw")
        }

        try {
            machineControllerService?.let {
                for (service in it) {
                    service.reset()
                }
            } ?: run {
                printLog("$TAG, Try To Reset Machine Service But Is Null")
            }
        } catch (ex: Exception){
            val sw = StringWriter()
            val pw = PrintWriter(sw)
            ex.printStackTrace(pw)
            printLog("$TAG: Exception Resetting Machine Service\n$sw")
        }

        try {
            coreService?.reload()
        } catch (ex: Exception){
            val sw = StringWriter()
            val pw = PrintWriter(sw)
            ex.printStackTrace(pw)
            printLog("$TAG: Exception Reloading Core Service\n$sw")
        }

        try {
            paymentService?.reload()
        } catch (ex: Exception){
            val sw = StringWriter()
            val pw = PrintWriter(sw)
            ex.printStackTrace(pw)
            printLog("$TAG: Exception Reloading Payment Service\n$sw")
        }

        try {
            machineControllerService?.reload()
        } catch (ex: Exception) {
            val sw = StringWriter()
            val pw = PrintWriter(sw)
            ex.printStackTrace(pw)
            printLog("$TAG: Exception Reloading Machine Controller Service\n$sw")
        }
    }

    fun addPaymentListener(listener: PaymentResultListener) {
        this.paymentListener = listener
        paymentService?.let {
            for (service in it) {
                service.addPaymentListener(listener)
            }
        }
    }

    /**
     * Load core jar
     *
     * @param context     only for android
     * @param filePath    path of jar
     * @return            ServiceLoader<CoreService>
     */
    private fun loadCoreJar(filePath: String): ServiceLoader<CoreService>? {
        printLog("$TAG: Loading Core Service")
        val classLoader = MyClassLoader(filePath)
        coreFileLocation = filePath
        return ServiceLoader.load(CoreService::class.java,classLoader)
    }

    /**
     * load payment jar
     *
     * @param context    only for android
     * @param filePath   path of jar
     * @return           ServiceLoader<PaymentService>
     */
    private fun loadPaymentJar(filePath: String): ServiceLoader<PaymentService>? {
        printLog("$TAG: Loading Payment Service")
        val classLoader = MyClassLoader(filePath)
        return ServiceLoader.load(PaymentService::class.java,classLoader)
    }

    /**
     * load machineController jar
     *
     * @param context     only for android
     * @param filePath    path of jar
     * @return            ServiceLoader<MachineControllerService>
     */
    private fun loadMachineControllerJar(filePath: String): ServiceLoader<MachineControllerService>? {
        printLog("$TAG: Loading Machine Controller Service")
        val classLoader = MyClassLoader(filePath)
        return ServiceLoader.load(MachineControllerService::class.java, classLoader)
    }

    fun setClientMessageProcessor(messageProcessor: MessageProcessor) {
        printLog("$TAG, Set Client Message Processor")
        this.clientMessageProcessor = messageProcessor
    }

    fun sendMessage(tag: String, msg: MessageBody){
        printLog("$TAG, Send Message To $tag")
        this.registeredMessageProcessor[tag]?.processMessage(msg) ?: run {
            printLog("$TAG, Cannot Find Message Processor For $tag")
        }
    }

    fun updateMachineState(machineState: MachineState, description: String, vararg stateCode: Pair<String,String>) {
        this.machineState = machineState
        this.description = description
        stateCodeList.clear()
        stateCodeList.addAll(stateCode)
        paymentService?.let {
            for (service in it) {
                service.updateMachineState(machineState,description,*stateCode)
            }
        }
    }

    fun setLogPath(path: String) {
        Utils.storeLog(path)
    }

    companion object {
        val TAG: String = BeepLoaderManager::class.java.simpleName

        private const val INTENDED_APP_VER = "intendedAppVer"
        private const val INTENDED_APP_PATH = "intendedAppPath"
        const val BOUNCY_CASTLE = "bouncy castle"
        @Volatile
        var paymentService: ServiceLoader<PaymentService>? = null
        @Volatile
        var machineControllerService: ServiceLoader<MachineControllerService>? = null
        @Volatile
        var coreService: ServiceLoader<CoreService>? = null

        @Volatile
        private var instance: BeepLoaderManager? = null

        fun getInstance(): BeepLoaderManager {
            return instance
                ?: synchronized(this) {
                return instance
                    ?: BeepLoaderManager()
                        .also { instance = it }
            }
        }
    }
}