package video.api.client.api.upload;

import video.api.client.api.ApiException;
import okhttp3.MediaType;
import okhttp3.RequestBody;
import okhttp3.internal.Util;
import okio.BufferedSink;
import okio.Okio;
import okio.Source;

import javax.annotation.Nullable;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.function.Consumer;

public class UploadChunkRequestBody extends RequestBody {

    private final InputStream inputStream;
    private final long chunkLength;
    private final String fileName;

    public UploadChunkRequestBody(File file, Integer chunksCount, Integer chunkNum, long totalBytes, long chunkStart,
            long chunkEnd, UploadProgressListener progressListener) throws ApiException {
        this.chunkLength = chunkEnd - chunkStart;
        this.fileName = file.getName();

        try {
            this.inputStream = new ChunkedInputStream(file, chunkStart, chunkEnd, progress -> {
                if (progressListener != null)
                    progressListener.onProgress(progress, totalBytes, chunksCount, chunkNum);
            });
        } catch (IOException e) {
            throw new ApiException(e);
        }
    }

    @Nullable
    @Override
    public MediaType contentType() {
        return MediaType.parse("video/mp4");
    }

    @Override
    public long contentLength() {
        return chunkLength;
    }

    @Override
    public void writeTo(BufferedSink sink) throws IOException {
        Source source = null;
        try {
            source = Okio.source(inputStream);
            sink.writeAll(source);

        } finally {
            Util.closeQuietly(source);
        }
    }

    public String getFileName() {
        return fileName;
    }

    private static class ChunkedInputStream extends InputStream {
        private final long start;
        private final InputStream source;
        private final long end;
        private final Consumer<Long> progressListener;
        private long pos = 0L;

        public ChunkedInputStream(File file, long start, long end, Consumer<Long> progressListener) throws IOException {
            this.source = new FileInputStream(file);
            this.start = start;
            this.end = end;
            this.progressListener = progressListener;

            progressListener.accept(start);

            source.skip(start);
        }

        @Override
        public int read() throws IOException {
            if (start + pos < end) {
                pos++;
                if (start + pos % 1024 == 0) {
                    progressListener.accept(start + pos);
                }
                return source.read();
            }

            progressListener.accept(start + pos);
            return -1;
        }
    }
}
