package org.codehaus.xfire.java;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.List;
import org.codehaus.plexus.component.repository.exception.ComponentLookupException;
import org.codehaus.xfire.SOAPConstants;
import org.codehaus.xfire.XFireRuntimeException;
import org.codehaus.xfire.fault.FaultHandler;
import org.codehaus.xfire.java.mapping.TypeMapping;
import org.codehaus.xfire.java.mapping.TypeMappingRegistry;
import org.codehaus.xfire.plexus.PlexusService;

/**
 * @author <a href="mailto:dan@envoisolutions.com">Dan Diephouse</a>
 */
public abstract class AbstractJavaService
    extends PlexusService
    implements JavaService
{
    private TypeMapping typeMapping;
    
    private List allowedMethods;
    
    private Class serviceClass;
    
    private Hashtable operations;
    
    protected static final String SERVICE_CLASS = "serviceClass";

    protected static final String ALLOWED_METHODS= "allowedMethods";

    private boolean autoTyped = false;
    
    public AbstractJavaService()
    {
        super();
        allowedMethods = new ArrayList();
        operations = new Hashtable();
    }

    /**
     * @param className
     * @param string
     */
    public void setServiceClass( String className )
        throws ClassNotFoundException
    {
        serviceClass = getClass().getClassLoader().loadClass( className );
        
        initializeOperations();
    }

    /**
     * @param serviceClass2
     */
    private void initializeOperations()
    {
        Method[] methods = serviceClass.getDeclaredMethods();
        
        // TODO: go through superclasses, stopping at Object.class
        
        for ( int i = 0; i < methods.length; i++ )
        {
            Method method = methods[i];
            
            String methodName = method.getName();
            
            int modifiers = method.getModifiers();
            if ( isAllowed( methodName )
                 &&
                 Modifier.isPublic(modifiers)
                 &&
                 !Modifier.isStatic(modifiers) )
            {
                addOperation( method );
            }
        }
    }

    /**
     * @param i
     * @param method
     * @param methodName
     */
    private void addOperation( Method method )
    {
        Operation op = new Operation(method, this);
        
        operations.put( method.getName(), op );
    }

    /**
     * Determines whether or not to expose the specified method.
     * 
     * @param methodName
     * @return
     */
    private boolean isAllowed(String methodName)
    {
        return ( allowedMethods.size() == 0 || allowedMethods.contains(methodName) );
    }

    /**
     * @see org.codehaus.xfire.java.JavaService#getOperation(java.lang.String, java.lang.String)
     */
    public Operation getOperation( String localName, String namespace )
    {
        return (Operation) operations.get( localName );
    }

    /**
     * @see org.codehaus.xfire.java.JavaService#getOperations()
     */
    public Collection getOperations()
    {
        return operations.values();
    }
    
    /**
     * @return Returns the allowedMethods.
     */
    public List getAllowedMethods()
    {
        return allowedMethods;
    }

    /**
     * @param allowedMethods The allowedMethods to set.
     */
    public void setAllowedMethods(List allowedMethods)
    {
        this.allowedMethods = allowedMethods;
    }

    /**
     * @return Returns the typeMapping.
     */
    public TypeMapping getTypeMapping()
    {
        return typeMapping;
    }

    /**
     * @param typeMapping The typeMapping to set.
     */
    public void setTypeMapping(TypeMapping typeMapping)
    {
        this.typeMapping = typeMapping;
    }

    /**
     * @return
     */
    public Class getServiceClass()
    {
        return serviceClass;
    }

    /**
     * Load a class from the class loader.
     * 
     * @param className The name of the class.
     * @return The class.
     * @throws Exception
     */
    protected Class loadClass( String className )
        throws Exception
    {
        // Handle array'd types.
        if ( className.endsWith("[]") )
        {
            className = "[L" + className.substring(0, className.length() - 2 ) + ";";
        }
        
        return getClass().getClassLoader().loadClass( className );
    }
    
    /**
     * @see org.apache.avalon.framework.activity.Initializable#initialize()
     */
    public void initialize() throws Exception
    {
        TypeMapping tm = null;
        
        // If we are a enocded, use the encoded type mapping, else just use XSD
        if ( getUse().equals(SOAPConstants.USE_ECNODED) )
        {
            tm = getTypeMappingRegistry().createTypeMapping( SOAPConstants.SOAP11_ENCODED, autoTyped );
        }
        else
        {
            tm = getTypeMappingRegistry().createTypeMapping( SOAPConstants.XSD, autoTyped );
        }
        
        setTypeMapping( tm );
        getTypeMappingRegistry().register( getDefaultNamespace(), tm );
        
        super.initialize();
    }
    
    public TypeMappingRegistry getTypeMappingRegistry()
    {
        try
        {
            return (TypeMappingRegistry) getServiceLocator().lookup( TypeMappingRegistry.ROLE );
        }
        catch (ComponentLookupException e)
        {
            throw new RuntimeException( "There is no type mapping registry!", e );
        }
    }
        
    public FaultHandler getFaultHandler()
    {
        try
        {
            return (FaultHandler)  getServiceLocator().lookup( FaultHandler.ROLE, getFaultHandlerHint() );
        }
        catch (ComponentLookupException e)
        {
            throw new XFireRuntimeException( "Couldn't find fault handler!", e );
        }
    }
 
	public boolean isAutoTyped()
	{
		return autoTyped;
	}
    
	public void setAutoTyped(boolean autoTyped)
	{
		this.autoTyped = autoTyped;
	}
}
