package org.codehaus.xfire.java;

import org.codehaus.plexus.configuration.PlexusConfiguration;
import org.codehaus.plexus.configuration.PlexusConfigurationException;
import org.codehaus.plexus.personality.plexus.lifecycle.phase.Configurable;
import org.codehaus.xfire.SOAPConstants;
import org.codehaus.xfire.fault.SOAP11FaultHandler;
import org.codehaus.xfire.fault.SOAP12FaultHandler;
import org.codehaus.xfire.handler.dom.DOMHandler;
import org.codehaus.xfire.handler.dom.DOMPipelineHandler;
import org.codehaus.xfire.java.mapping.TypeMapping;
import org.codehaus.xfire.wsdl.WSDLBuilder;
import org.dom4j.QName;

/**
 * A service that is created from an XML configuration within Plexus.
 * 
 * @author <a href="mailto:dan@envoisolutions.com">Dan Diephouse</a>
 */
public class XmlJavaService
	extends AbstractJavaService
	implements Configurable
{
    private PlexusConfiguration[] types;
    
    private PlexusConfiguration[] handlers;
    
    /**
     * @see org.apache.avalon.framework.configuration.Configurable#configure(org.apache.avalon.framework.configuration.Configuration)
     */
    public void configure(PlexusConfiguration config) throws PlexusConfigurationException
    {
        configureService(config);
        
        configureTypes(config);
        
        DOMPipelineHandler handler = new DOMPipelineHandler();
        setServiceHandler( handler );
        
        handlers = config.getChild("handlers").getChildren("handler");
    }

    /**
     * @param config
     * @throws PlexusConfigurationException
     */
    private void configureService(PlexusConfiguration config) throws PlexusConfigurationException
    {
        setName( config.getChild("name").getValue() );
        
        setDefaultNamespace( config.getChild( "namespace" ).getValue("") );
    
        setWSDLURL( config.getChild("wsdlURL").getValue("") );
        
        setUse( config.getChild("use").getValue("literal") );
        
        setStyle( config.getChild("style").getValue("wrapped") );
        
        try
        {
            setServiceClass( config.getChild( SERVICE_CLASS ).getValue() );
        }
        catch (ClassNotFoundException e)
        {
            throw new PlexusConfigurationException( "Couldn't find service class.", e );
        }
        
        // TODO use allowed methods attribute
        setProperty( ALLOWED_METHODS, config.getChild( ALLOWED_METHODS ).getValue("") );
        
        String soapNS = config.getChild( "soapVersion" ).getValue("1.1");
        
        if ( soapNS.equals("1.1") )
        {
            setSoapVersion( SOAPConstants.SOAP11_ENVELOPE_NS );
            setFaultHandlerHint( SOAP11FaultHandler.NAME );
        }
        else if ( soapNS.equals("1.2") )
        {
            setFaultHandlerHint( SOAP12FaultHandler.NAME );
            setSoapVersion( SOAPConstants.SOAP12_ENVELOPE_NS );
        }
        else
            throw new PlexusConfigurationException("Invalid soap version.  Must be 1.1 or 1.2.");
            
        setFaultHandlerHint( soapNS );
        
        setAutoTyped( Boolean.valueOf(config.getChild( "autoTyped" ).getValue("false")).booleanValue() );
    }

    /**
     * @param config
     * @throws PlexusConfigurationException
     */
    private void configureTypes(PlexusConfiguration config) throws PlexusConfigurationException
    {
       types  = config.getChild("types").getChildren("type");
    }
    
    public void initialize() throws Exception
    {
        super.initialize();
        
        DOMPipelineHandler pipe = (DOMPipelineHandler) getServiceHandler();

        if ( handlers != null && handlers.length > 0 )
        {
            for ( int i = 0; i < handlers.length; i++ )
            {
                String hId = handlers[i].getValue();
                
                DOMHandler handler = (DOMHandler) getServiceLocator().lookup( DOMHandler.ROLE, hId );
                
                pipe.getHandlers().add( handler );
            }
        }
        else
        {
            pipe.getHandlers().add( new JavaServiceHandler() );
        }
        
        for ( int i = 0; i < types.length; i++ )
        {
            initializeType( types[i], getTypeMapping() );   
        }
        
        
        setWSDLBuilder( (WSDLBuilder) getServiceLocator().lookup(WSDLBuilder.ROLE, "java") );
    }

    private void initializeType(PlexusConfiguration configuration, TypeMapping tm) throws PlexusConfigurationException
    {
        try
        {
            String ns = configuration.getAttribute( "namespace", getDefaultNamespace() );
            String name = configuration.getAttribute("name");
            
            tm.register( loadClass( configuration.getAttribute("class") ),
                         QName.get( name, ns ),
                         loadClass( configuration.getAttribute("type") ) );
        }
        catch (Exception e)
        {
            if ( e instanceof PlexusConfigurationException )
                throw (PlexusConfigurationException) e;
            
            throw new PlexusConfigurationException( "Could not configure type.", e );
        }                     
    }
}
