package org.codehaus.xfire.java.mapping;

import java.util.Collection;
import java.util.StringTokenizer;

import org.codehaus.xfire.java.type.ArrayType;
import org.codehaus.xfire.java.type.BeanType;
import org.codehaus.xfire.java.type.Type;
import org.dom4j.QName;


/**
 * A type mapping which automatically generates types
 * for java classes which are not registered, allowing
 * easy deployment of java services.
 * 
 * @author <a href="mailto:dan@envoisolutions.com">Dan Diephouse</a>
 * @since Feb 21, 2004
 */
public class AutoTypeMapping
    extends CustomTypeMapping
{

    public AutoTypeMapping( TypeMapping defaultTM )
    {
        super( defaultTM );
    }
    
    public AutoTypeMapping()
    {
        super();
    }
	
	/**
	 * @see org.codehaus.xfire.java.mapping.TypeMapping#getSerializer(java.lang.Class, javax.xml.namespace.QName)
	 */
	public Type getType( Class javaType, QName xmlType )
	{
        Type type = super.getType(javaType, xmlType);
        
        if ( type == null )
        {
        	register( javaType, xmlType, findTypeClass( javaType ) );
            
            type = super.getType(javaType, xmlType);
        }
        
        return type;
	}

    /**
     * Tries to determine a type class automatically from the type.
     * 
	 * @param javaType
	 * @return
	 */
	private Class findTypeClass(Class javaType)
	{
		if ( javaType.isArray() ||
             javaType.isAssignableFrom( Collection.class ) )
        {
			return ArrayType.class;
        }
        else
        {
        	return BeanType.class;
        }
	}

	/**
     * @see org.codehaus.xfire.java.mapping.TypeMapping#getType(java.lang.Class)
     */
    public Type getType(Class javaType)
    {
        Type type = super.getType(javaType);
        
        if ( type == null )
        {
            QName qname = createQName( javaType );
            register( javaType, qname, findTypeClass( javaType ) );

            type = super.getType(javaType, qname);
        }
        
        return type;
    }

	/**
	 * @param javaType
	 * @return
	 */
	private QName createQName(Class javaType)
	{
        String clsName = javaType.getName();
        
        if (clsName.startsWith("[L"))
		{
			clsName = clsName.substring(2, clsName.length() - 1);
		}
        
        String ns = AutoTypeMapping.makeNamespaceFromClassName(clsName, "http");
        
        String localName = null;
        
        if (javaType.isArray() ||
            javaType.isAssignableFrom( Collection.class ))
        {
            localName = "ArrayOf" + clsName.substring( clsName.lastIndexOf(".")+1 );
            
            // If this is an array of a primitive type, put the type
            // we're creating in the default namespace.
            if ( javaType.isArray() )
            {
                Type type = getType( javaType.getComponentType() );
                
                ns = getEncodingStyleURI();
            }
        }
        else
        {
            localName = clsName.substring( clsName.lastIndexOf(".")+1 );
        }

        return QName.get( localName, ns );
	}

    private static String makeNamespaceFromClassName(String className, String protocol)
	{
        String packageName = className.substring(0, className.lastIndexOf("."));
        
		if ((packageName == null) || packageName.equals(""))
		{
			return protocol + "://" + "DefaultNamespace";
		}

		StringTokenizer st = new StringTokenizer(packageName, ".");
		String[] words = new String[st.countTokens()];

		for (int i = 0; i < words.length; ++i)
		{
			words[i] = st.nextToken();
		}

		StringBuffer sb = new StringBuffer(80);

		for (int i = words.length - 1; i >= 0; --i)
		{
			String word = words[i];

			// seperate with dot
			if (i != words.length - 1)
			{
				sb.append('.');
			}

			sb.append(word);
		}

		return protocol + "://" + sb.toString();
	}
    
}
