package org.codehaus.xfire.java.mapping;

import java.util.HashMap;
import java.util.Map;
import org.codehaus.xfire.java.type.Type;
import org.dom4j.QName;

/**
 * Contains type mappings for java/qname pairs and 
 * Serializer/Deserializer factories.
 * 
 * @author <a href="mailto:dan@envoisolutions.com">Dan Diephouse</a>
 * @since Feb 21, 2004
 */
public class CustomTypeMapping
    implements TypeMapping
{
	private Map types;
    
    private Map class2Type;

    private Map xml2Type;

    private Map class2xml;
    
    private TypeMapping defaultTM;
    
    private String encodingStyleURI;
    
    public CustomTypeMapping( TypeMapping defaultTM )
    {
        this();
        
        this.defaultTM = defaultTM;
    }
    
    public CustomTypeMapping()
    {
        types = new HashMap();
        class2Type = new HashMap();
        class2xml = new HashMap();
        xml2Type = new HashMap();
    }

	/**
	 * @see org.codehaus.xfire.java.mapping.TypeMapping#isRegistered(java.lang.Class, javax.xml.namespace.QName)
	 */
	public boolean isRegistered( Class javaType, QName xmlType )
	{
        ClassQNamePair pair = new ClassQNamePair(javaType, xmlType);
        
        boolean registered =  types.containsKey( pair );
        
        if ( !registered && defaultTM != null )
            registered = defaultTM.isRegistered( javaType, xmlType );
        
        return registered;
	}

	/**
	 * @see org.codehaus.xfire.java.mapping.TypeMapping#register(java.lang.Class, javax.xml.namespace.QName, org.codehaus.xfire.java.encoding.SerializerFactory, org.codehaus.xfire.java.encoding.DeserializerFactory)
	 */
	public void register( Class javaType, QName xmlType, Class typeClass )
	{
		ClassQNamePair pair = new ClassQNamePair(javaType, xmlType);
        
		types.put( pair, typeClass );
        
        class2Type.put( javaType, typeClass );
        xml2Type.put( xmlType, typeClass );
        class2xml.put( javaType, xmlType );
	}

	/**
	 * @see org.codehaus.xfire.java.mapping.TypeMapping#getSerializer(java.lang.Class, javax.xml.namespace.QName)
	 */
	public Type getType( Class javaType, QName xmlType )
	{
        ClassQNamePair pair = new ClassQNamePair(javaType, xmlType);
        
        Type type = instantiateType( (Class) types.get( pair ),
                                     javaType,
                                     xmlType );

        if ( type == null && defaultTM != null )
            type = defaultTM.getType( javaType, xmlType );
        
        return type;
	}
    
	/**
	 * @see org.codehaus.xfire.java.mapping.TypeMapping#removeSerializer(java.lang.Class, javax.xml.namespace.QName)
	 */
	public void removeType( Class javaType, QName xmlType )
	{
        ClassQNamePair pair = new ClassQNamePair(javaType, xmlType);
        
        Type type = instantiateType( (Class) types.remove( pair ), javaType, xmlType );

        if ( type == null && defaultTM != null )
           defaultTM.removeType( javaType, xmlType );
	}

    /**
     * @see org.codehaus.xfire.java.mapping.TypeMapping#getType(java.lang.Class)
     */
    public Type getType(Class javaType)
    {
        Type type = instantiateType( (Class) class2Type.get( javaType ), 
                                     javaType,
                                     (QName) getTypeQName( javaType ) );

        if ( type == null && defaultTM != null )
            type = defaultTM.getType( javaType );
        
        return type;
    }

    /**
     * @see org.codehaus.xfire.java.mapping.TypeMapping#getType(javax.xml.namespace.QName)
     */
    public Type getType(QName xmlType)
    {
        Type type = (Type) xml2Type.get( xmlType );

        if ( type == null && defaultTM != null )
            type = defaultTM.getType( xmlType );
        
        return type;
    }

    /**
     * @see org.codehaus.xfire.java.mapping.TypeMapping#getTypeQName(java.lang.Class)
     */
    public QName getTypeQName(Class clazz)
    {
        QName qname = (QName) class2xml.get( clazz );

        if ( qname == null && defaultTM != null )
            qname = defaultTM.getTypeQName( clazz );
        
        return qname;
    }
    
    public Type instantiateType( Class typeClass, Class clazz, QName qname )
    {
		try
		{
            Type type = (Type) typeClass.newInstance();
            type.setTypeClass( clazz );
            type.setSchemaType( qname );
            type.setTypeMapping( this );
            
            return type;
		}
		catch (Exception e)
		{
            return null;
		}
    }
    
    public String getEncodingStyleURI()
    {
        return encodingStyleURI;
    }
    
    public void setEncodingStyleURI( String encodingStyleURI )
    {
        this.encodingStyleURI = encodingStyleURI;
    }
}
