package org.codehaus.xfire.java.mapping;

import java.util.Hashtable;
import java.util.Iterator;
import org.codehaus.plexus.configuration.PlexusConfiguration;
import org.codehaus.plexus.configuration.PlexusConfigurationException;
import org.codehaus.plexus.personality.plexus.lifecycle.phase.Configurable;
import org.codehaus.xfire.SOAPConstants;
import org.codehaus.xfire.java.type.BooleanType;
import org.codehaus.xfire.java.type.DoubleType;
import org.codehaus.xfire.java.type.FloatType;
import org.codehaus.xfire.java.type.IntType;
import org.codehaus.xfire.java.type.LongType;
import org.codehaus.xfire.plexus.PlexusXFireComponent;
import org.dom4j.QName;

/**
 * The default implementation of TypeMappingRegistry.
 * 
 * @author <a href="mailto:dan@envoisolutions.com">Dan Diephouse</a>
 * @since Feb 22, 2004
 */
public class DefaultTypeMappingRegistry
    extends PlexusXFireComponent
    implements TypeMappingRegistry, Configurable
{
    private Hashtable registry;
    
    private TypeMapping defaultTM;
    
    public DefaultTypeMappingRegistry()
    {
        registry = new Hashtable();
    }
    
	/**
	 * @see org.codehaus.xfire.java.mapping.TypeMappingRegistry#register(java.lang.String, org.codehaus.xfire.java.encoding.TypeMapping)
	 */
	public TypeMapping register( String encodingStyleURI, TypeMapping mapping )
	{
        TypeMapping previous = (TypeMapping) registry.get( encodingStyleURI );
       
        mapping.setEncodingStyleURI( encodingStyleURI );
        
		registry.put( encodingStyleURI, mapping );
        
        return previous;
	}

    /**
	 * @see org.codehaus.xfire.java.mapping.TypeMappingRegistry#registerDefault(org.codehaus.xfire.java.encoding.TypeMapping)
	 */
	public void registerDefault( TypeMapping mapping )
	{
		defaultTM = mapping;
	}
    
	/**
	 * @see org.codehaus.xfire.java.mapping.TypeMappingRegistry#getDefaultTypeMapping()
	 */
	public TypeMapping getDefaultTypeMapping()
	{
		return defaultTM;
	}

    /**
	 * @see org.codehaus.xfire.java.mapping.TypeMappingRegistry#getRegisteredEncodingStyleURIs()
	 */
	public String[] getRegisteredEncodingStyleURIs()
	{
		return (String[]) registry.keySet().toArray(new String[0]);
	}

    /**
	 * @see org.codehaus.xfire.java.mapping.TypeMappingRegistry#getTypeMapping(java.lang.String)
	 */
	public TypeMapping getTypeMapping( String encodingStyleURI )
	{
		return (TypeMapping) registry.get(encodingStyleURI);
	}
    
	/**
	 * @see org.codehaus.xfire.java.mapping.TypeMappingRegistry#createTypeMapping()
	 */
	public TypeMapping createTypeMapping( boolean autoTypes )
	{
		return createTypeMapping( getDefaultTypeMapping(), autoTypes );
	}

    /**
     * @see org.codehaus.xfire.java.mapping.TypeMappingRegistry#createTypeMapping(java.lang.String)
     */
    public TypeMapping createTypeMapping( String parentNamespace, boolean autoTypes )
    {
        return createTypeMapping( getTypeMapping( parentNamespace ), autoTypes );
    }
    
    protected TypeMapping createTypeMapping( TypeMapping parent, boolean autoTypes )
    {
        if ( autoTypes )
        {
        	return new AutoTypeMapping( parent );
        }
        else
        {
        	return new CustomTypeMapping( parent );
        }
    }
    
    /**
	 * @see org.codehaus.xfire.java.mapping.TypeMappingRegistry#unregisterTypeMapping(java.lang.String)
	 */
	public TypeMapping unregisterTypeMapping( String encodingStyleURI )
	{
		TypeMapping tm = (TypeMapping) registry.get(encodingStyleURI);
        registry.remove(encodingStyleURI);
        return tm;
	}
    
	/**
	 * @see org.codehaus.xfire.java.mapping.TypeMappingRegistry#removeTypeMapping(org.codehaus.xfire.java.encoding.TypeMapping)
	 */
	public boolean removeTypeMapping( TypeMapping mapping )
	{
        int n = 0;
        
        for ( Iterator itr = registry.values().iterator(); itr.hasNext(); )
        {
            if ( itr.next().equals( mapping ) )
            {
            	itr.remove();
                n++;
            }
        }
        
        return (n > 0);
	}

	/**
	 * @see org.codehaus.xfire.java.mapping.TypeMappingRegistry#clear()
	 */
	public void clear()
	{
		registry.clear();
	}

    /**
     * @see org.apache.avalon.framework.configuration.Configurable#configure(org.apache.avalon.framework.configuration.Configuration)
     */
    public void configure(PlexusConfiguration config) throws PlexusConfigurationException
    {
        PlexusConfiguration tmConfig[] = config.getChildren("typeMapping");
        
        for ( int i = 0; i < tmConfig.length; i++ )
        {
            configureTypeMapping( tmConfig[i] );
        }
    }

    /**
     * @param configuration
     */
    private void configureTypeMapping(PlexusConfiguration configuration)
        throws PlexusConfigurationException
    {
        TypeMapping tm = createTypeMapping(false);
        
        register( configuration.getAttribute( "namespace" ), tm );
        
        if ( Boolean.valueOf( configuration.getAttribute("default", "false") ).booleanValue() )
            registerDefault( tm );
        
        PlexusConfiguration[] types = configuration.getChildren( "type" );
        
        // register primitive types manually since there is no way
        // to do Class.forName("boolean") et al.
        tm.register(boolean.class, QName.get("boolean", SOAPConstants.XSD), BooleanType.class);
        tm.register(int.class, QName.get("int", SOAPConstants.XSD), IntType.class);
        tm.register(double.class, QName.get("double", SOAPConstants.XSD), DoubleType.class);
        tm.register(float.class, QName.get("float", SOAPConstants.XSD), FloatType.class);
        tm.register(long.class, QName.get("long", SOAPConstants.XSD), LongType.class);
        
        for ( int i = 0; i < types.length; i++ )
        {
            configureType( types[i], tm );
        }
    }
    
    private void configureType( PlexusConfiguration configuration, TypeMapping tm )
        throws PlexusConfigurationException
    {
        try
        {
            String ns = configuration.getAttribute("namespace");
            String name = configuration.getAttribute("name");
            QName qname = QName.get( name, ns );
            
            Class clazz = loadClass( configuration.getAttribute("class") );
            Class typeClass = loadClass( configuration.getAttribute("type") );

            tm.register( clazz,
                         qname,
                         typeClass );
            
            getLogger().debug( "Registered " + typeClass.getName() + 
                              " for " + qname + " with class " + clazz.getName() );
        }
        catch (Exception e)
        {
            if ( e instanceof PlexusConfigurationException )
                throw (PlexusConfigurationException) e;
            
            throw new PlexusConfigurationException( "Could not configure type.", e );
        }                     
    }
    
    /**
     * Load a class from the class loader.
     * 
     * @param className The name of the class.
     * @return The class.
     * @throws Exception
     */
    private Class loadClass( String className )
        throws Exception
    {
        // Handle array'd types.
        if ( className.endsWith("[]") )
        {
            className = "[L" + className.substring(0, className.length() - 2 ) + ";";
        }
        
        return getClass().getClassLoader().loadClass( className );
    }
}
