package cn.openjava.basic.format.vo;

import cn.hutool.json.JSONUtil;
import cn.openjava.basic.format.message.IMessage;
import cn.openjava.basic.format.message.MessageType;
import cn.openjava.basic.format.message.SystemMessage;
import com.fasterxml.jackson.annotation.JsonFormat;
import lombok.Builder;
import lombok.Getter;
import org.springframework.format.annotation.DateTimeFormat;

import java.io.Serializable;
import java.text.MessageFormat;
import java.util.Date;
import java.util.UUID;

/**
 * 响应结果
 *
 * @param <T> 任意类型
 */
@Getter
@Builder
public class Result<T> implements Serializable {

    private static final long serialVersionUID = -8864618211140193686L;
    /**
     * UUID
     */
    private String uuid;

    /**
     * 结果类型
     */
    private MessageType type;

    /**
     * 状态码
     */
    private int code;

    /**
     * 消息
     */
    private String msg;

    /**
     * 结果值
     */
    private T data;

    /**
     * 时间戳
     */
    @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
    @JsonFormat(pattern = "yyyy-MM-dd HH:mm:ss", timezone = "GMT+8")
    private Date time;

    /**
     * 加密
     */
    private int crypto;

    public static <T> Result data(T data) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).code(SystemMessage.SUCCESS.code()).data(data).build();
    }

    /**
     * 成功
     *
     * @param dialog 对话框
     * @return 操作结果
     */
    public static Result success(IMessage dialog) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(dialog.message()).code(SystemMessage.SUCCESS.code()).type(MessageType.SUCCESS).build();
    }

    /**
     * 成功
     *
     * @param msg 消息
     * @return 操作结果
     */
    public static Result success(String msg) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(msg).code(SystemMessage.SUCCESS.code()).type(MessageType.SUCCESS).build();
    }

    public static <T> Result success(T msg) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(JSONUtil.toJsonStr(msg)).code(SystemMessage.SUCCESS.code()).type(MessageType.SUCCESS).build();
    }

    public static <T> Result success(T data, IMessage dialog) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(dialog.message()).data(data).code(SystemMessage.SUCCESS.code()).type(MessageType.SUCCESS).build();
    }

    public static Result warning(String message) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(message).code(SystemMessage.NORMAL_WARNING.code()).type(MessageType.WARNING).build();
    }


    public static <T> Result success(T data, IMessage dialog, Object... fillStr) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(MessageFormat.format(dialog.message(), fillStr)).data(data).code(SystemMessage.SUCCESS.code()).type(MessageType.SUCCESS).build();
    }

    public static <T> Result success(T data, int crypto, IMessage dialog, Object... fillStr) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(MessageFormat.format(dialog.message(), fillStr)).data(data).code(SystemMessage.SUCCESS.code()).crypto(crypto).type(MessageType.SUCCESS).build();
    }

    /**
     * 消息-返回数据
     *
     * @param data 数据
     * @param <T>  泛型，任意类型
     * @return result对象
     */
    public static <T> Result info(T data) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).data(data).code(SystemMessage.INFO.code()).type(MessageType.INFO).build();
    }


    /**
     * 失败
     *
     * @param iDialog 页面会话
     * @return 结果
     */
    public static Result warning(IMessage iDialog) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(iDialog.message()).code(iDialog.code()).type(MessageType.WARNING).build();
    }

    public static Result warning(IMessage iDialog, Object... fillStr) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(MessageFormat.format(iDialog.message(), fillStr)).code(iDialog.code()).type(MessageType.WARNING).build();
    }

    /**
     * 失败
     *
     * @param msg  错误消息
     * @param code 错误码
     * @return 结果
     */
    public static Result warning(String msg, int code) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(msg).code(code).type(MessageType.WARNING).build();
    }

    /**
     * 失败
     *
     * @param exMessage 错误消息
     * @return 结果
     */
    public static Result error(String exMessage) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(exMessage).code(SystemMessage.EXCEPTION.code()).type(MessageType.ERROR).build();
    }

    /**
     * 失败
     *
     * @param iDialog 页面会话
     * @return 结果
     */
    public static Result error(IMessage iDialog) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(iDialog.message()).code(iDialog.code()).type(MessageType.ERROR).build();
    }

    public static Result error(IMessage iDialog, Object... fillStr) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(MessageFormat.format(iDialog.message(), fillStr)).code(iDialog.code()).type(MessageType.ERROR).build();
    }

    /**
     * 消息-不确定是SUCCESS还是WARNING还是ERROR
     *
     * @param iDialog
     * @return
     */
    public static Result message(IMessage iDialog) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(iDialog.message()).code(iDialog.code()).type(iDialog.type()).build();
    }

    public static Result message(MessageType type, int code, String msg) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(msg).code(code).type(type).build();
    }

    /**
     * 消息-不确定是SUCCESS还是WARNING还是ERROR
     *
     * @param iDialog
     * @return
     */
    public static Result message(IMessage iDialog, Object... fillStr) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(MessageFormat.format(iDialog.message(), fillStr)).code(iDialog.code()).type(iDialog.type()).build();
    }

    /**
     * 失败
     *
     * @param eMsg 错误消息
     * @param code 错误码
     * @return 结果
     */
    public static Result error(String eMsg, int code) {
        return Result.builder().uuid(UUID.randomUUID().toString()).time(new Date()).msg(eMsg).code(code).type(MessageType.ERROR).build();
    }

    public Result setType(MessageType type) {
        this.type = type;
        return this;
    }

    public Result setData(T data) {
        this.data = data;
        return this;
    }

    public Result setCrypto(int crypto) {
        this.crypto = crypto;
        return this;
    }

    public Result setMsg(String msg) {
        this.msg = msg;
        return this;
    }
}
