package cn.schoolwow.data.thread;

import cn.schoolwow.data.thread.config.QuickDataThreadConfig;
import cn.schoolwow.data.thread.domain.DataThreadProgress;
import cn.schoolwow.data.thread.flow.handler.DataThreadBeforeAfterHandler;
import cn.schoolwow.data.thread.flow.work.*;
import cn.schoolwow.data.thread.flow.work.map.GetMapFileFlow;
import cn.schoolwow.data.thread.work.interval.IntervalDataThreadWork;
import cn.schoolwow.data.thread.work.loop.LoopDataThreadWork;
import cn.schoolwow.data.thread.work.map.MapDataThreadWork;
import cn.schoolwow.data.thread.work.map.MapWorkResult;
import cn.schoolwow.data.thread.work.reduce.ReduceDataThreadWork;
import cn.schoolwow.data.thread.work.single.SingleDataThreadWork;
import cn.schoolwow.quickflow.QuickFlow;
import cn.schoolwow.quickflow.domain.FlowContext;
import org.apache.commons.io.FileUtils;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;
import java.util.List;

/**多线程工具类*/
public class QuickDataThread {
    private QuickFlow dataThreadFlow = QuickFlow.newInstance();

    private QuickDataThreadConfig quickDataThreadConfig = new QuickDataThreadConfig(dataThreadFlow);

    /**实例化*/
    public static QuickDataThread newInstance() {
        return new QuickDataThread();
    }

    /**配置信息*/
    public QuickDataThreadConfig config(){
        return quickDataThreadConfig;
    }

    /**获取进度列表*/
    public List<DataThreadProgress> getDataThreadProgressList(){
        List<DataThreadProgress> dataThreadProgressList = (List<DataThreadProgress>) dataThreadFlow.getContextData("progressList");
        return dataThreadProgressList;
    }

    /**中断任务*/
    public void interruptDataThread(long id){
        List<DataThreadProgress> dataThreadProgressList = getDataThreadProgressList();
        for(DataThreadProgress dataThreadProgress:dataThreadProgressList){
            if(dataThreadProgress.id==id){
                dataThreadProgress.status = "中断结束";
                dataThreadProgress.threadPoolExecutor.shutdownNow();
            }
        }
    }

    /**删除进度信息*/
    public void deleteDataThreadProgress(long id){
        List<DataThreadProgress> dataThreadProgressList = getDataThreadProgressList();
        Iterator<DataThreadProgress> iterator = dataThreadProgressList.iterator();
        while(iterator.hasNext()){
            DataThreadProgress dataThreadProgress = iterator.next();
            if(dataThreadProgress.id==id){
                //判断是否已经结束或者抛出异常
                if(null!=dataThreadProgress.endTime||null!=dataThreadProgress.exception){
                    iterator.remove();
                }else{
                    throw new IllegalArgumentException("只允许删除任务正常结束或者异常结束的数据线程!进度id:"+id);
                }
            }
        }
    }

    /**执行Interval类型任务*/
    public void executeIntervalDataThreadWork(IntervalDataThreadWork intervalDataThreadWork){
        QuickFlow workFlow = intervalDataThreadWork.getWorkFlow();
        workFlow.startFlow(new ExecuteIntervalWorkFlow<>())
                .beforeAfterFlowHandler(new DataThreadBeforeAfterHandler())
                .putCurrentCompositeFlowData("dataThreadFlow", dataThreadFlow)
                .execute();
    }

    /**执行Loop类型任务*/
    public void executeLoopDataThreadWork(LoopDataThreadWork loopDataThreadWork){
        QuickFlow workFlow = loopDataThreadWork.getWorkFlow();
        workFlow.startFlow(new ExecuteLoopWorkFlow<>())
                .beforeAfterFlowHandler(new DataThreadBeforeAfterHandler())
                .putCurrentCompositeFlowData("dataThreadFlow", dataThreadFlow)
                .execute();
    }

    /**执行Single类型任务*/
    public void executeSingleDataThreadWork(SingleDataThreadWork singleDataThreadWork){
        QuickFlow workFlow = singleDataThreadWork.getWorkFlow();
        workFlow.startFlow(new ExecuteSingleWorkFlow<>())
                .beforeAfterFlowHandler(new DataThreadBeforeAfterHandler())
                .putCurrentCompositeFlowData("dataThreadFlow", dataThreadFlow)
                .execute();
    }

    /**判断map文件是否存在*/
    public boolean existMapFile(String name, String id) {
        File file = (File) dataThreadFlow.startFlow(new GetMapFileFlow())
                .putCurrentCompositeFlowData("workName", name)
                .putCurrentCompositeFlowData("id", id)
                .execute()
                .checkData("mapFile");
        return file.exists();
    }

    /**获取mapFile文件内容*/
    public String getMapFileContent(String name, String id) throws IOException {
        File file = (File) dataThreadFlow.startFlow(new GetMapFileFlow())
                .putCurrentCompositeFlowData("workName", name)
                .putCurrentCompositeFlowData("id", id)
                .execute()
                .checkData("mapFile");
        return FileUtils.readFileToString(file, StandardCharsets.UTF_8);
    }

    /**执行Map类型任务*/
    public MapWorkResult executeMapDataThreadWork(MapDataThreadWork mapDataThreadWork){
        QuickFlow workFlow = mapDataThreadWork.getWorkFlow();
        FlowContext flowContext = workFlow.startFlow(new ExecuteMapWorkFlow<>())
                .beforeAfterFlowHandler(new DataThreadBeforeAfterHandler())
                .putCurrentCompositeFlowData("dataThreadFlow", dataThreadFlow)
                .execute();
        MapWorkResult mapWorkResult = new MapWorkResult();
        mapWorkResult.name = (String) flowContext.checkData("name");
        mapWorkResult.idList = (List<String>) flowContext.getData("idList");
        return mapWorkResult;
    }

    /**执行Map类型任务*/
    public void executeReduceMapDataThreadWork(ReduceDataThreadWork reduceDataThreadWork) throws IOException {
        QuickFlow workFlow = reduceDataThreadWork.getWorkFlow();
        FlowContext flowContext = workFlow.startFlow(new ExecuteReduceWorkFlow<>())
                .beforeAfterFlowHandler(new DataThreadBeforeAfterHandler())
                .putCurrentCompositeFlowData("dataThreadFlow", dataThreadFlow)
                .execute();
        boolean deleteFile = (boolean) flowContext.getData("deleteFile", true);
        if(deleteFile){
            String mapDirectory = (String) flowContext.checkData("mapDirectory");
            FileUtils.forceDelete(new File(mapDirectory));
        }
    }

}
