package co.cloudcraft;

import co.cloudcraft.api.AwsAccountApi;
import co.cloudcraft.api.BlueprintApi;
import co.cloudcraft.api.BudgetApi;
import co.cloudcraft.api.UserApi;
import co.cloudcraft.exception.CloudcraftException;
import lombok.AccessLevel;
import lombok.Getter;

@Getter
public class Cloudcraft {
  private static final String CLOUDCRAFT_PROTOCOL = "CLOUDCRAFT_PROTOCOL";
  private static final String CLOUDCRAFT_HOST = "CLOUDCRAFT_HOST";
  private static final String CLOUDCRAFT_PORT = "CLOUDCRAFT_PORT";
  private static final String CLOUDCRAFT_API_KEY = "CLOUDCRAFT_API_KEY";
  private static final String CLOUDCRAFT_TIMEOUT = "CLOUDCRAFT_TIMEOUT";
  private static final String CLOUDCRAFT_MAX_NETWORK_RETRIES = "CLOUDCRAFT_MAX_NETWORK_RETRIES";

  /** Client configuration */
  private ClientConfig clientConfig;

  /** Key used to access cloudcraft APIs */
  private final String apiKey;

  @Getter(AccessLevel.NONE)
  private final RestClient restClient;

  private final UserApi userApi;

  private final AwsAccountApi awsAccountApi;

  private final BlueprintApi blueprintApi;

  private final BudgetApi budgetApi;

  /**
   * Initializes client configuration from environment variables if present, else fallback to
   * default values.
   *
   * @throws CloudcraftException if API Key is not configured.
   */
  public Cloudcraft() throws CloudcraftException {
    this(null, null);
  }

  /**
   * Initializes with the given APIKey and initializes client configuration from environment
   * variables if present, else fallback to default values.
   *
   * @throws CloudcraftException if API Key is not configured.
   */
  public Cloudcraft(String apiKey) throws CloudcraftException {
    this(apiKey, null);
  }

  /**
   * Initializes API key and client configuration with the given overrides.
   *
   * @throws CloudcraftException if API Key is not configured.
   */
  public Cloudcraft(String apiKey, ClientConfig clientConfigOverrides) throws CloudcraftException {
    this.apiKey = apiKey != null ? apiKey : System.getenv(CLOUDCRAFT_API_KEY);
    if (this.apiKey == null) {
      throw new CloudcraftException("No API key specified", "400");
    }

    initializeClientConfig(clientConfigOverrides);
    restClient = new RestClient(this.apiKey, clientConfig);
    userApi = new UserApi(restClient);
    awsAccountApi = new AwsAccountApi(restClient);
    blueprintApi = new BlueprintApi(restClient);
    budgetApi = new BudgetApi(restClient);
  }

  /**
   * Initialize Cloudcraft SDK client with the following order - If an explicit value is provided in
   * the <code>clientConfigOverrides</code> use that - If no override is provided, extract it from
   * configured environment variable - If no env variables is configured, use the default value
   *
   * @param clientConfigOverrides overridding values for client configuration
   */
  private void initializeClientConfig(ClientConfig clientConfigOverrides) {
    String protocol =
        (clientConfigOverrides != null && clientConfigOverrides.getProtocol() != null)
            ? clientConfigOverrides.getProtocol()
            : System.getenv(CLOUDCRAFT_PROTOCOL) != null
                ? System.getenv(CLOUDCRAFT_PROTOCOL)
                : ClientConfig.DEFAULT_PROTOCOL;

    String host =
        (clientConfigOverrides != null && clientConfigOverrides.getHost() != null)
            ? clientConfigOverrides.getHost()
            : System.getenv(CLOUDCRAFT_HOST) != null
                ? System.getenv(CLOUDCRAFT_HOST)
                : ClientConfig.DEFAULT_HOST;

    Integer port =
        (clientConfigOverrides != null && clientConfigOverrides.getPort() != null)
            ? clientConfigOverrides.getPort()
            : System.getenv(CLOUDCRAFT_PORT) != null
                ? Integer.parseInt(System.getenv(CLOUDCRAFT_PORT))
                : ClientConfig.DEFAULT_PORT;

    Integer maxRetries =
        (clientConfigOverrides != null && clientConfigOverrides.getMaxRetries() != null)
            ? clientConfigOverrides.getMaxRetries()
            : System.getenv(CLOUDCRAFT_MAX_NETWORK_RETRIES) != null
                ? Integer.parseInt(System.getenv(CLOUDCRAFT_MAX_NETWORK_RETRIES))
                : ClientConfig.DEFAULT_MAX_NETWORK_RETRIES;

    Integer readTimeout =
        (clientConfigOverrides != null && clientConfigOverrides.getReadTimeout() != null)
            ? clientConfigOverrides.getReadTimeout()
            : System.getenv(CLOUDCRAFT_TIMEOUT) != null
                ? Integer.parseInt(System.getenv(CLOUDCRAFT_TIMEOUT))
                : ClientConfig.DEFAULT_TIMEOUT_MS;

    this.clientConfig = new ClientConfig(protocol, host, port, readTimeout, maxRetries);
  }
}
