package co.cloudcraft;

import co.cloudcraft.exception.*;
import co.cloudcraft.model.CloudcraftObject;
import co.cloudcraft.model.CloudcraftResponse;
import com.google.gson.JsonSyntaxException;
import java.io.IOException;
import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import lombok.NonNull;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.HttpClientBuilder;
import org.apache.hc.core5.http.*;
import org.apache.hc.core5.http.io.entity.EmptyInputStream;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.apache.hc.core5.http.io.support.ClassicRequestBuilder;
import org.apache.hc.core5.util.Timeout;

// TODO: connection pooling, http client configuration setting
public class RestClient {
  private final String apiKey;

  private final String baseUrl;

  CloseableHttpClient httpClient;

  public RestClient(String apiKey, ClientConfig config) {
    this.apiKey = apiKey;

    RequestConfig.Builder requestConfigBuilder = RequestConfig.copy(RequestConfig.DEFAULT);
    requestConfigBuilder.setConnectTimeout(
        Timeout.of(config.getReadTimeout(), TimeUnit.MILLISECONDS));
    requestConfigBuilder.setConnectionRequestTimeout(
        Timeout.of(config.getReadTimeout(), TimeUnit.MILLISECONDS));
    RequestConfig httpConfig = requestConfigBuilder.build();
    HttpClientBuilder builder = HttpClientBuilder.create();
    builder.setDefaultRequestConfig(httpConfig);

    httpClient = builder.build();

    String apiBasePath =
        System.getenv("CLOUDCRAFT_BASE_PATH") != null
            ? System.getenv("CLOUDCRAFT_BASE_PATH")
            : ClientConfig.DEFAULT_BASE_PATH;
    this.baseUrl =
        String.format(
            "%s://%s:%d%s", config.getProtocol(), config.getHost(), config.getPort(), apiBasePath);
  }

  public CloudcraftResponse execute(Method method, String path) throws CloudcraftException {
    return execute(method, path, null, null, null);
  }

  public CloudcraftResponse execute(Method method, String path, Map<String, String> params)
      throws CloudcraftException {
    return execute(method, path, null, null, params);
  }

  public CloudcraftResponse execute(
      Method method, String path, Map<String, String> headers, String requestBodyJson)
      throws CloudcraftException {
    return execute(method, path, headers, requestBodyJson, null);
  }

  CloudcraftResponse execute(
      Method method,
      String path,
      Map<String, String> headers,
      String requestBodyJson,
      Map<String, String> params)
      throws CloudcraftException {
    try {
      ClassicRequestBuilder reqBuilder =
          ClassicRequestBuilder.create(method.toString()).setUri(constructFullPath(path));
      addHeaders(reqBuilder, headers);
      addParameters(reqBuilder, params);

      if (requestBodyJson != null) {
        reqBuilder.setEntity(requestBodyJson, ContentType.APPLICATION_JSON);
      }

      ClassicHttpResponse httpResponse = httpClient.execute(reqBuilder.build());
      HttpEntity httpEntity = httpResponse.getEntity();

      int responseCode = httpResponse.getCode();
      String responseContentType = getHeader(httpResponse.getHeaders(), HttpHeaders.CONTENT_TYPE);
      // handle non 2xx conditions
      if (responseCode < 200 || responseCode >= 300) {
        try {
          String responseBodyAsString = EntityUtils.toString(httpResponse.getEntity());
          handleErrorResponse(responseCode, responseBodyAsString, responseContentType);
        } catch (ParseException pe) {
          handleParseException(responseCode, "Unable to parse response", pe);
        }
      }

      return new CloudcraftResponse(
          httpResponse.getCode(),
          httpEntity != null ? httpEntity.getContent() : EmptyInputStream.INSTANCE,
          responseContentType,
          getHeader(httpResponse.getHeaders(), HttpHeaders.ETAG));
    } catch (IOException ioe) {
      throw new CloudcraftException(ioe.getMessage(), "500", 500, ioe);
    }
  }

  void handleErrorResponse(int responseCode, String responseBody, String contentType)
      throws CloudcraftException {
    CloudcraftException exception;

    try {
      ErrorResponse error;
      if (contentType.startsWith("application/json")) {
        error =
            responseBody != null
                ? CloudcraftObject.GSON.fromJson(responseBody, ErrorResponse.class)
                : new ErrorResponse("No additional information", responseCode);
      } else {
        error = new ErrorResponse(responseBody, responseCode);
      }

      switch (responseCode) {
        case HttpStatus.SC_CLIENT_ERROR:
          exception = new InvalidRequestException(error);
          break;
        case HttpStatus.SC_NOT_FOUND:
          exception = new NotFoundException(error);
          break;
        case HttpStatus.SC_FORBIDDEN:
          exception = new PermissionException(error);
          break;
        case HttpStatus.SC_UNAUTHORIZED:
          exception = new AuthenticationException(error);
          break;
        case HttpStatus.SC_TOO_MANY_REQUESTS:
          exception = new RateLimitException(error);
          break;
        case HttpStatus.SC_PRECONDITION_FAILED:
          exception = new ResourceOutofDateException(error);
          break;
        default:
          exception = new CloudcraftException(error, responseCode);
          break;
      }
      throw exception;
    } catch (JsonSyntaxException e) {
      handleParseException(responseCode, responseBody, e);
    }
  }

  void handleParseException(int responseCode, String responseBody, Throwable e)
      throws CloudcraftException {
    String details = e == null ? "none" : e.getMessage();
    String apiResponseBody = responseBody == null ? "unparsable" : responseBody;
    throw new CloudcraftException(
        String.format(
            "Invalid response object from API: %s. (HTTP response code was %d). Additional details: %s.",
            apiResponseBody, responseCode, details),
        Integer.toString(responseCode),
        responseCode,
        e);
  }

  void addHeaders(ClassicRequestBuilder reqBuilder, Map<String, String> headers) {
    // add authentication header
    reqBuilder.addHeader("Authorization", "Bearer " + this.apiKey);

    // add other headers if specified
    if (headers != null && !headers.isEmpty()) {
      for (Map.Entry<String, String> entry : headers.entrySet()) {
        reqBuilder.addHeader(entry.getKey(), entry.getValue());
      }
    }
  }

  void addParameters(ClassicRequestBuilder reqBuilder, Map<String, String> params) {
    if (params != null && !params.isEmpty()) {
      for (Map.Entry<String, String> entry : params.entrySet()) {
        reqBuilder.addParameter(entry.getKey(), entry.getValue());
      }
    }
  }

  String constructFullPath(@NonNull String path) {
    return this.baseUrl + path;
  }

  public String getHeader(@NonNull Header[] headers, @NonNull String headerName) {
    if (headers == null) {
      return null;
    }

    Optional<Header> header =
        Arrays.stream(headers).filter(h -> h.getName().equalsIgnoreCase(headerName)).findFirst();
    return header.map(NameValuePair::getValue).orElse(null);
  }
}
