package co.cloudcraft.api;

import co.cloudcraft.RestClient;
import co.cloudcraft.exception.CloudcraftException;
import co.cloudcraft.model.*;
import java.util.Map;
import lombok.NonNull;
import org.apache.hc.core5.http.Method;

public class AwsAccountApi extends ApiBase {

  public AwsAccountApi(RestClient restClient) {
    super(restClient);
  }

  /**
   * List all the parameters required for registering a new IAM Role in AWS for use with Cloudcraft,
   * customized to you.
   *
   * <p>This API, combined with the AWS CLI to generate IAM Roles, can facilitate fully automated
   * role creation at scale for an organization with a lot of AWS accounts.
   *
   * @return
   * @throws CloudcraftException when any error occurs exercising this API
   */
  public IAMParameters getCloudcraftIAMRoleConfig() throws CloudcraftException {
    CloudcraftResponse response = restClient.execute(Method.GET, "/aws/account/iamParameters");
    return parseCloudcraftResponse(response, IAMParameters.class);
  }

  /** Register a new AWS account with Cloudcraft, for visualization via the API */
  public AwsAccount add(@NonNull AwsAccountRequest accountRequest) throws CloudcraftException {
    accountRequest.validate();
    CloudcraftResponse response =
        restClient.execute(
            Method.POST, "/aws/account", CONTENT_TYPE_HEADER_MAP, accountRequest.toJson());
    return parseCloudcraftResponse(response, AwsAccount.class);
  }

  /** Update a registered AWS account by ID */
  public AwsAccount update(@NonNull String accountId, @NonNull AwsAccountRequest accountRequest)
      throws CloudcraftException {
    accountRequest.validate();

    CloudcraftResponse response =
        restClient.execute(
            Method.PUT,
            String.format("/aws/account/%s", accountId),
            CONTENT_TYPE_HEADER_MAP,
            accountRequest.toJson());
    return parseCloudcraftResponse(response, AwsAccount.class);
  }

  /** Delete a registered AWS account by ID */
  public void delete(@NonNull String accountId) throws CloudcraftException {
    restClient.execute(Method.DELETE, String.format("/aws/account/%s", accountId));
  }

  /**
   * List all of your linked AWS accounts. The response is an array of AWS accounts. Each entry
   * includes the account ID and name, access control and user information.
   *
   * @return
   * @throws CloudcraftException when any error occurs exercising this API
   */
  public AwsAccountCollection list() throws CloudcraftException {
    CloudcraftResponse response = restClient.execute(Method.GET, "/aws/account");
    return parseCloudcraftResponse(response, AwsAccountCollection.class);
  }

  /**
   * Scan and render one region of an AWS account into a blueprint in JSON, SVG, PNG, PDF or MxGraph
   * format.
   *
   * <p>The time required to generate the snapshot depends on the number of resources in the AWS
   * region. The API behaves as a long poll, with a wait time of up to 120 seconds for the result.
   * For most environments the API call will therefore directly return a blueprint. In case the wait
   * time is exceeded, a 202 Accepted response is returned with a {code: STILL_PROCESSING, retry:
   * true ...} JSON body. The snapshot will continue processing in the background, and a retry will
   * either immediately return the result or continue waiting.
   *
   * @param accountId Cloudcraft issued identifier for the registered AWS account.
   * @param region the AWS region
   * @param format expected response format
   * @param requestParams optional parameters
   * @return CloudcraftResponse object
   * @throws CloudcraftException when any error occurs exercising this API
   */
  public CloudcraftResponse snapshot(
      @NonNull String accountId,
      @NonNull String region,
      @NonNull ApiBase.BlueprintExportFormat format,
      SnapshotAccountQueryParams requestParams)
      throws CloudcraftException {
    Map<String, String> paramsMap = requestParams != null ? requestParams.toMap() : null;

    return restClient.execute(
        Method.GET, String.format("/aws/account/%s/%s/%s", accountId, region, format), paramsMap);
  }
}
