package co.cloudcraft.api;

import co.cloudcraft.RestClient;
import co.cloudcraft.exception.CloudcraftException;
import co.cloudcraft.model.*;
import java.util.HashMap;
import java.util.Map;
import lombok.NonNull;
import org.apache.hc.core5.http.HttpHeaders;
import org.apache.hc.core5.http.Method;

public class BlueprintApi extends ApiBase {

  public BlueprintApi(RestClient restClient) {
    super(restClient);
  }

  /**
   * List all of your saved blueprints.
   *
   * <p>The response is an array of blueprint entries without the diagram data. Each entry includes
   * the blueprint ID and name, access control and user information.
   *
   * @return List of blueprint entries without the diagram data
   * @throws CloudcraftException when any error occurs exercising this API
   */
  public BlueprintCollection list() throws CloudcraftException {
    CloudcraftResponse response = restClient.execute(Method.GET, "/blueprint");
    return parseCloudcraftResponse(response, BlueprintCollection.class);
  }

  /**
   * Create a new blueprint. The response will contain the created blueprint, including the newly
   * assigned ID
   *
   * @param blueprintRequest Blueprint document to be created.
   * @return Created blueprint with additional metadata.
   * @throws CloudcraftException when any error occurs exercising this API
   */
  public Blueprint create(@NonNull BlueprintRequest blueprintRequest) throws CloudcraftException {
    CloudcraftResponse response =
        restClient.execute(
            Method.POST, "/blueprint", CONTENT_TYPE_HEADER_MAP, blueprintRequest.toJson());
    return parseCloudcraftResponse(response, Blueprint.class);
  }

  /**
   * Retrieve one blueprint by ID
   *
   * @throws CloudcraftException when any error occurs exercising this API
   */
  public Blueprint get(@NonNull String blueprintId) throws CloudcraftException {
    CloudcraftResponse response =
        restClient.execute(Method.GET, String.format("/blueprint/%s", blueprintId));
    return parseCloudcraftResponse(response, Blueprint.class);
  }

  /**
   * Update an existing blueprint by ID
   *
   * @throws CloudcraftException when any error occurs exercising this API
   */
  public void update(@NonNull String blueprintId, @NonNull BlueprintRequest blueprintRequest)
      throws CloudcraftException {
    updateWithEtag(blueprintId, blueprintRequest, null);
  }

  /**
   * Update an existing blueprint by ID and the same ETag value as provided by the "Retrieve
   * blueprint" API. If the blueprint has been modified since the retrieval, the update will be
   * rejected with a 412 Resource out of date response.
   *
   * @throws CloudcraftException when any error occurs exercising this API
   */
  public void updateWithEtag(
      @NonNull String blueprintId, @NonNull BlueprintRequest blueprintRequest, String eTag)
      throws CloudcraftException {
    Map<String, String> headers = new HashMap<>(CONTENT_TYPE_HEADER_MAP);
    if (eTag != null) {
      headers.put(HttpHeaders.IF_MATCH, eTag);
    }
    restClient.execute(
        Method.PUT,
        String.format("/blueprint/%s", blueprintId),
        headers,
        blueprintRequest.toJson());
  }

  /**
   * Delete an existing blueprint by ID
   *
   * @throws CloudcraftException when any error occurs exercising this API
   */
  public void delete(@NonNull String blueprintId) throws CloudcraftException {
    restClient.execute(Method.DELETE, String.format("/blueprint/%s", blueprintId));
  }

  /**
   * Render blueprint for export in SVG, PNG, PDF or MxGraph format. Consumer of this API can
   * extract the response using the convenience APIs in <code>CloudcraftResponse</code> either in
   * <code>String</code> or <code>byte[]</code> or <code>JSON</code> format.
   *
   * @param blueprintId blueprint identifier
   * @param format export format
   * @param requestParams optional parameters that
   * @return CloudcraftResponse object
   * @throws CloudcraftException when any error occurs exercising this API
   */
  public CloudcraftResponse exportBlueprint(
      @NonNull String blueprintId,
      @NonNull ApiBase.BlueprintExportFormat format,
      ExportBlueprintQueryParams requestParams)
      throws CloudcraftException {
    Map<String, String> paramsMap = requestParams != null ? requestParams.toMap() : null;

    return restClient.execute(
        Method.GET, String.format("/blueprint/%s/%s", blueprintId, format), paramsMap);
  }
}
