package co.cloudcraft.model;

import com.google.gson.annotations.SerializedName;
import lombok.*;

/** Data object representing AWS account registered with Cloudcraft. */
@RequiredArgsConstructor
@Getter
@EqualsAndHashCode(callSuper = false)
@ToString
public class AwsAccount extends CloudcraftObject {
  /** Cloudcraft issued unique identifier. */
  private final String id;

  /** Human readable name for the AWS account */
  private final String name;

  /**
   * The ARN of the read-only IAM Role created in your AWS account for Cloudcraft. For instructions
   * on creating the role, please see the "AWS Accounts" dialog in the web application.
   */
  private final String roleArn;

  /**
   * An external ID to uniquely associate with the role. For the purposes of the Developer API, the
   * IAM Role must be created with the unique external ID value of the person who generated the API
   * key being used.
   */
  private final String externalId;

  /** List of teams or customers with read access. */
  private final String[] readAccess;

  /** List of teams or customers with write access. */
  private final String[] writeAccess;

  /** Cloudcraft issued identifier of the user that registered this account. */
  @SerializedName("CreatorId")
  private final String creatorId;

  @Getter(AccessLevel.NONE)
  private final String createdAt;

  @Getter(AccessLevel.NONE)
  private final String updatedAt;

  /**
   * Returns when the user account was last updated.
   *
   * @return updated time in milliseconds since epoch.
   */
  public Long getCreatedTime() {
    return convertDateStringToInstant(this.createdAt);
  }

  /**
   * Returns when the user last access Cloudcraft.
   *
   * @return last accessed time in milliseconds since epoch.
   */
  public Long getUpdatedTime() {
    return convertDateStringToInstant(this.updatedAt);
  }
}
