package co.cloudcraft.model;

import co.cloudcraft.exception.ErrorResponse;
import co.cloudcraft.exception.InvalidRequestException;
import com.google.gson.annotations.SerializedName;
import lombok.*;

/** Request object representing the inputs required to register an AWS Account with Cloudcraft. */
@AllArgsConstructor
@RequiredArgsConstructor
@Getter
@Setter
@EqualsAndHashCode(callSuper = false)
@ToString
public class AwsAccountRequest extends CloudcraftObject {
  /** Human readable name for the AWS account */
  @NonNull private String name;

  /**
   * The ARN of the read-only IAM Role created in your AWS account for Cloudcraft. For instructions
   * on creating the role, please see the "AWS Accounts" dialog in the web application.
   */
  @NonNull private String roleArn;

  /**
   * Represents the AWS region to be used for account validation. By default, the account will be
   * validated in the "us-east-1" region.
   */
  @SerializedName("region")
  private String accountValidationRegion;

  /** List of teams or customers with read access. */
  private String[] readAccess;

  /** List of teams or customers with write access. */
  private String[] writeAccess;

  public AwsAccountRequest(AwsAccount account) {
    this.name = account.getName();
    this.roleArn = account.getRoleArn();
    this.readAccess = account.getReadAccess();
    this.writeAccess = account.getWriteAccess();
  }

  /**
   * This method does a sanity check of the requests object and ensure the inputs are present and
   * valid.
   *
   * @throws InvalidRequestException
   */
  public void validate() throws InvalidRequestException {
    if (name.trim().isEmpty()) {
      throw new InvalidRequestException(new ErrorResponse("Missing required parameter: name", 400));
    }

    if (roleArn.trim().isEmpty()) {
      throw new InvalidRequestException(
          new ErrorResponse("Missing required parameter: roleArn", 400));
    }

    String[] segments = roleArn.trim().split(":");
    if (segments.length < 6 || !roleArn.startsWith("arn")) {
      throw new InvalidRequestException(
          new ErrorResponse(String.format("Invalid roleArn specified: %s", roleArn), 400));
    }
  }
}
