package co.cloudcraft.model;

import co.cloudcraft.exception.CloudcraftException;
import com.google.gson.JsonObject;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.stream.Collectors;
import lombok.AllArgsConstructor;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.ToString;

@AllArgsConstructor
@Getter
@ToString
@EqualsAndHashCode(callSuper = false)
public class CloudcraftResponse {
  /** HTTP response code */
  Integer statusCode;
  /** Response content InputStream */
  InputStream responseBody;

  /** Content type of the response */
  String contentType;

  /** HTTP ETag header, if any, returned in the response */
  String eTag;

  /**
   * Parse and return the response as a String. Response can be converted to String if the
   * content-type is one of 1. application/json 2. application/xml 3. image/svg+xml 4. text/csv
   *
   * @return response content as String.
   * @throws CloudcraftException when any error occurs exercising this API
   */
  public String getResponseAsString() throws CloudcraftException {
    if (canConvertToString()) {
      return new BufferedReader(new InputStreamReader(responseBody, StandardCharsets.UTF_8))
          .lines()
          .collect(Collectors.joining("\n"));
    } else {
      throw new CloudcraftException(
          String.format("Response cannot be converted to String: content-type:%s", contentType),
          "400");
    }
  }

  /**
   * Parse and return the response as a JSON.
   *
   * @return response content as JsonObject.
   * @throws CloudcraftException when any error occurs exercising this API
   */
  public JsonObject getResponseAsJson() throws CloudcraftException {
    if (!contentType.startsWith("application/json")) {
      throw new CloudcraftException(
          String.format(
              "Response cannot be converted to JSON Object: content-type:%s", contentType),
          "400");
    }

    BufferedReader reader =
        new BufferedReader(new InputStreamReader(responseBody, StandardCharsets.UTF_8));
    return CloudcraftObject.GSON.fromJson(reader, JsonObject.class);
  }

  /**
   * Parse and return response as byte[].
   *
   * @return response content as byte[].
   */
  public byte[] getResponseAsBytes() throws IOException {
    if (responseBody == null) {
      return null;
    }

    ByteArrayOutputStream buffer = new ByteArrayOutputStream();

    int nRead;
    byte[] data = new byte[4096];

    while ((nRead = responseBody.read(data, 0, data.length)) != -1) {
      buffer.write(data, 0, nRead);
    }

    buffer.flush();
    return buffer.toByteArray();
  }

  private boolean canConvertToString() {
    return (contentType.startsWith("application/json")
        || contentType.startsWith("application/xml")
        || contentType.startsWith("image/svg+xml")
        || contentType.startsWith("text/csv"));
  }
}
