/*
 * Decompiled with CFR 0.152.
 */
package com.bitmark.sdk.features;

import com.bitmark.sdk.crypto.Ed25519;
import com.bitmark.sdk.crypto.Random;
import com.bitmark.sdk.crypto.SecretBox;
import com.bitmark.sdk.crypto.Sha3256;
import com.bitmark.sdk.crypto.encoder.Base58;
import com.bitmark.sdk.crypto.encoder.Hex;
import com.bitmark.sdk.crypto.key.KeyPair;
import com.bitmark.sdk.crypto.key.PublicKey;
import com.bitmark.sdk.error.ValidateException;
import com.bitmark.sdk.service.configuration.GlobalConfiguration;
import com.bitmark.sdk.service.configuration.Network;
import com.bitmark.sdk.utils.AccountNumberData;
import com.bitmark.sdk.utils.Address;
import com.bitmark.sdk.utils.ArrayUtil;
import com.bitmark.sdk.utils.RecoveryPhrase;
import com.bitmark.sdk.utils.Seed;
import com.bitmark.sdk.utils.Validator;

public class Account {
    private static final byte[] KEY_INDEX = Hex.HEX.decode("000000000000000000000000000003E7");
    private static final int CORE_LENGTH = 32;
    private static final int NONCE_LENGTH = 24;
    private String accountNumber;
    private byte[] core;

    public static Account fromSeed(Seed seed) throws ValidateException {
        Validator.checkValid(() -> seed.getNetwork() == GlobalConfiguration.network(), "Incorrect network from Seed");
        byte[] core = seed.getSeed();
        KeyPair key = Account.generateKeyPair(core);
        String accountNumber = Account.generateAccountNumber(key.publicKey(), seed.getNetwork());
        return new Account(core, accountNumber);
    }

    public static Account fromRecoveryPhrase(String ... recoveryPhrase) throws ValidateException {
        RecoveryPhrase phrase = RecoveryPhrase.fromMnemonicWords(recoveryPhrase);
        Seed seed = phrase.recoverSeed();
        return Account.fromSeed(seed);
    }

    public Account() {
        this.core = Random.randomBytes(32);
        KeyPair key = Account.generateKeyPair(this.core);
        this.accountNumber = Account.generateAccountNumber(key.publicKey());
    }

    private Account(byte[] core, String accountNumber) {
        this.core = core;
        this.accountNumber = accountNumber;
    }

    public KeyPair getKey() {
        return Account.generateKeyPair(this.core);
    }

    public String getAccountNumber() {
        return this.accountNumber;
    }

    public RecoveryPhrase getRecoveryPhrase() {
        return RecoveryPhrase.fromSeed(this.getSeed());
    }

    public Seed getSeed() {
        AccountNumberData data = Account.parseAccountNumber(this.accountNumber);
        return new Seed(this.core, data.getNetwork(), 1);
    }

    public Address toAddress() {
        return Address.fromAccountNumber(this.accountNumber);
    }

    public static boolean isValidAccountNumber(String accountNumber) {
        try {
            Account.parseAccountNumber(accountNumber);
            return true;
        }
        catch (ValidateException ex) {
            return false;
        }
    }

    public static AccountNumberData parseAccountNumber(String accountNumber) {
        Address address = Address.fromAccountNumber(accountNumber);
        return AccountNumberData.from(address.getKey(), address.getNetwork());
    }

    private static KeyPair generateKeyPair(byte[] core) {
        byte[] seed = SecretBox.generateSecretBox(KEY_INDEX, new byte[24], core);
        return Ed25519.generateKeyPairFromSeed(seed);
    }

    private static String generateAccountNumber(PublicKey key) {
        return Account.generateAccountNumber(key, GlobalConfiguration.network());
    }

    private static String generateAccountNumber(PublicKey key, Network network) {
        Address address = Address.getDefault(key, network);
        byte[] keyVariantVarInt = address.getPrefix();
        byte[] publicKeyBytes = key.toBytes();
        byte[] preChecksum = ArrayUtil.concat(keyVariantVarInt, publicKeyBytes);
        byte[] checksum = ArrayUtil.slice(Sha3256.hash(preChecksum), 0, 4);
        return Base58.BASE_58.encode(ArrayUtil.concat(keyVariantVarInt, publicKeyBytes, checksum));
    }
}

