/*******************************************************************************
 * Copyright (c) 2014, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client;

import java.util.Set;

import com.pushtechnology.diffusion.client.features.UpdateConstraint;
import com.pushtechnology.diffusion.client.features.control.Metrics;
import com.pushtechnology.diffusion.client.features.control.Metrics.SessionMetricCollector;
import com.pushtechnology.diffusion.client.features.control.Metrics.TopicMetricCollector;
import com.pushtechnology.diffusion.client.features.control.RemoteServers;
import com.pushtechnology.diffusion.client.features.control.RemoteServers.RemoteServer;
import com.pushtechnology.diffusion.client.features.control.RemoteServers.RemoteServer.RemoteServerBuilder;
import com.pushtechnology.diffusion.client.features.control.topics.SessionTrees;
import com.pushtechnology.diffusion.client.features.control.topics.SessionTrees.BranchMappingTable;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.session.SessionFactory;
import com.pushtechnology.diffusion.client.session.SessionId;
import com.pushtechnology.diffusion.client.session.proxy.ProxyAuthenticationFactory;
import com.pushtechnology.diffusion.client.topics.TopicSelectors;
import com.pushtechnology.diffusion.client.topics.details.TopicSpecification;
import com.pushtechnology.diffusion.client.topics.details.TopicType;
import com.pushtechnology.diffusion.client.types.Credentials;
import com.pushtechnology.diffusion.datatype.DataTypes;

/**
 * Provides facilities that are shared between sessions.
 *
 * @author DiffusionData Limited
 * @since 5.0
 */
public enum Diffusion {

    /** The implementation. */
    GLOBALS;

    private Implementation delegate;

    Diffusion() {

        final String implementation =
            "com.pushtechnology.diffusion.client.impl.SessionGlobals";

        try {
            @SuppressWarnings("unchecked")
            final Class<? extends Implementation> clazz =
                (Class<? extends Implementation>) this.getClass()
                    .getClassLoader().loadClass(implementation);

            delegate = clazz.newInstance();
        }
        catch (
            ClassNotFoundException | IllegalAccessException |
            InstantiationException e) {
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * Returns the shared session factory.
     *
     * @return session factory.
     */
    public static SessionFactory sessions() {
        return GLOBALS.delegate.getSessionFactory();
    }

    /**
     * Returns shared credentials factory to use for creating credentials.
     *
     * @return credentials factory.
     */
    public static Credentials.Factory credentials() {
        return GLOBALS.delegate.getCredentialsFactory();
    }

    /**
     * Returns a parser for topic selector expressions.
     *
     * @return topic selectors parser
     */
    public static TopicSelectors topicSelectors() {
        return GLOBALS.delegate.getTopicSelectors();
    }

    /**
     * Returns a proxy authentication scheme factory.
     *
     * @return proxy authentication scheme factory
     */
    public static ProxyAuthenticationFactory proxyAuthentication() {
        return GLOBALS.delegate.getProxyAuthenticationFactory();
    }

    /**
     * Returns data type implementations.
     *
     * @since 5.7
     */
    public static DataTypes dataTypes() {
        return GLOBALS.delegate.getDataTypes();
    }

    /**
     * Escapes special characters in a string that is to be used within a topic
     * property or a session filter.
     * <P>
     * This is a convenience method which inserts an escape character '\' before
     * any of the special characters ' " or \.
     *
     * @param string the string to be escaped
     *
     * @return the string value with escape characters inserted as appropriate
     *
     * @since 6.1
     */
    public static String escape(String string) {
        return GLOBALS.delegate.escape(string);
    }

    /**
     * Create a new {@link TopicSpecification} for a given topic type.
     *
     * @param topicType the topic type
     *
     * @return a new immutable specification with no properties set. New
     *         specifications with different properties can be produced using
     *         the {@link TopicSpecification#withProperty(String, String)
     *         withProperty} or
     *         {@link TopicSpecification#withProperties(java.util.Map)
     *         withProperties} methods of the provided specification.
     *
     * @since 6.2
     */
    public static TopicSpecification
        newTopicSpecification(TopicType topicType) {
        return GLOBALS.delegate.newTopicSpecification(topicType);
    }

    /**
     * Returns an update constraint factory.
     *
     * @return update constraint factory
     *
     * @see UpdateConstraint
     *
     * @since 6.2
     */
    public static UpdateConstraint.Factory updateConstraints() {
        return GLOBALS.delegate.updateConstraints();
    }

    /**
     * Utility method which converts a string of the format required by the
     * {@link Session#ROLES $Roles} session property into a mutable set of
     * strings.
     *
     * @param string the string with quoted roles separated by whitespace or
     *        commas
     *
     * @return set of roles
     *
     * @since 6.2
     */
    public static Set<String> stringToRoles(String string) {
        return GLOBALS.delegate.stringToRoles(string);
    }

    /**
     * Utility method which converts a set of authorisation roles to the string
     * format required by the {@link Session#ROLES $Roles} session property.
     *
     * @param roles a set of roles
     *
     * @return a string representing the supplied roles, formatted as required
     *         by the {@link Session#ROLES $Roles} session property
     *
     * @since 6.2
     */
    public static String rolesToString(Set<String> roles) {
        return GLOBALS.delegate.rolesToString(roles);
    }

    /**
     * Create a new {@link BranchMappingTable.Builder}.
     *
     * @see SessionTrees
     *
     * @since 6.7
     */
    public static BranchMappingTable.Builder newBranchMappingTableBuilder() {
        return GLOBALS.delegate.newBranchMappingTableBuilder();
    }

    /**
     * Create a new {@link SessionMetricCollector.Builder}.
     *
     * @see Metrics
     *
     * @since 6.7
     */
    public static SessionMetricCollector.Builder newSessionMetricCollectorBuilder() {
        return GLOBALS.delegate.newSessionMetricCollectorBuilder();
    }

    /**
     * Parse a {@link SessionId} from a string.
     * <p>
     * This method allows a {@code SessionId} to parsed from a string in the
     * format as generated by {@link SessionId#toString()}. This is the
     * representation used in the {@link Session#SESSION_ID SESSION_ID} session
     * property.
     *
     * @param sessionIdAsString a string, in the format as generated by
     *        {@link SessionId#toString()}
     *
     * @return the {@code SessionId}
     *
     * @throws IllegalArgumentException if the string was not compatible with
     *         the format as generated by {@link SessionId#toString()}
     *
     * @since 6.7
     */
    public static SessionId sessionIdFromString(String sessionIdAsString)
        throws IllegalArgumentException {
        return GLOBALS.delegate.sessionIdFromString(sessionIdAsString);
    }

    /**
     * Create a new {@link TopicMetricCollector.Builder}.
     *
     * @see Metrics
     *
     * @since 6.7
     */
    public static TopicMetricCollector.Builder newTopicMetricCollectorBuilder() {
        return GLOBALS.delegate.newTopicMetricCollectorBuilder();
    }

    /**
     * Create a new {@link RemoteServer.Builder}.
     *
     * @see RemoteServers
     *
     * @since 6.7
     *
     * @deprecated since 6.9
     *             <p>
     *             This method may only be used to create an old style builder
     *             for secondary initiators. The newer
     *             {@link #newRemoteServerBuilder(Class)} specifying the
     *             secondary initiator builder class should be used in
     *             preference. This method will be removed in a future release.
     */
    @Deprecated
    public static RemoteServer.Builder newRemoteServerBuilder() {
        return GLOBALS.delegate.newRemoteServerBuilder();
    }

    /**
     * Create a new {@link RemoteServerBuilder}.
     *
     * @param builderInterface the builder interface
     *
     * @param <B> builder type
     *
     * @return the builder
     *
     * @throws IllegalArgumentException if {@code builderInterface} is not a
     *         supported builder interface
     *
     * @see RemoteServers
     *
     * @since 6.9
     */
    public static <B extends RemoteServerBuilder<B>> B newRemoteServerBuilder(
        Class<B> builderInterface)
        throws IllegalArgumentException {
        return GLOBALS.delegate.newRemoteServerBuilder(builderInterface);
    }

    /**
     * The contract with the implementation.
     *
     * <p>
     * This is for internal use only.
     */
    public interface Implementation {
        /**
         * @return implementation
         */
        SessionFactory getSessionFactory();

        /**
         * @return implementation
         */
        Credentials.Factory getCredentialsFactory();

        /**
         * @return implementation
         */
        TopicSelectors getTopicSelectors();

        /**
         * @return implementation
         */
        ProxyAuthenticationFactory getProxyAuthenticationFactory();

        /**
         * @return implementation
         */
        DataTypes getDataTypes();

        /**
         * @return implementation
         */
        TopicSpecification newTopicSpecification(TopicType topicType);

        /**
         * @return implementation
         */
        UpdateConstraint.Factory updateConstraints();

        /**
         * Parses a string with roles to a mutable set.
         */
        Set<String> stringToRoles(String string);

        /**
         * Formats set of roles into a string.
         */
        String rolesToString(Set<String> roles);

        /**
         * Escapes special characters in a string.
         */
        String escape(String string);

        /**
         * @return new instance
         */
        BranchMappingTable.Builder newBranchMappingTableBuilder();

        /**
         * @return new instance
         */
        SessionMetricCollector.Builder newSessionMetricCollectorBuilder();

        /**
         * @return new instance
         */
        TopicMetricCollector.Builder newTopicMetricCollectorBuilder();

        /**
         * @return new instance
         */
        RemoteServer.Builder newRemoteServerBuilder();

        /**
         * @return new instance
         */
        <B extends RemoteServerBuilder<B>> B newRemoteServerBuilder(
            Class<B> builderInterface)
            throws IllegalArgumentException;

        /**
         * Parses a session id from a string.
         */
        SessionId sessionIdFromString(String sessionIdAsString)
            throws IllegalArgumentException;
    }
}
