/*******************************************************************************
 * Copyright (c) 2014, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.features;

import static org.slf4j.LoggerFactory.getLogger;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import org.slf4j.Logger;

import com.pushtechnology.diffusion.client.session.Feature;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.session.SessionClosedException;

/**
 * This feature provides a client session with the ability to test its
 * connection to the server.
 *
 * <P>
 * The main purpose of a ping is to test, at a very basic level, the current
 * network conditions that exist between the client session and the server it is
 * connected to. The ping response includes the time taken to make a round-trip
 * call to the server.
 *
 * <H3>Access control</H3>
 *
 * <p>
 * There are no permissions requirements associated with this feature.
 *
 * <H3>Accessing the feature</H3> This feature may be obtained from a
 * {@link Session session}, and used as follows:
 *
 * <pre>
 * Pings pings = session.feature(Pings.class);
 *
 * CompletableFuture&lt;PingDetails&gt; response = pings.pingServer();
 *
 * PingDetails result = response.get();  // Block for response
 *
 * System.out.printf("Round-trip call to server took %d milliseconds%n", result.getRoundTripTime());
 * </pre>
 *
 * <p>Alternatively, the CompletableFuture API can be used to process the result
 * using the Diffusion input thread, without blocking the calling thread:
 *
 * <pre>
 * Pings pings = session.feature(Pings.class);
 *
 * pings.pingServer()
 *      .thenApply(PingDetails::getRoundTripTime)
 *      .thenAccept(t -&gt; System.out.printf("Round-trip call to server took %d milliseconds%n", t));
 * </pre>
 *
 * @author DiffusionData Limited
 * @since 5.0
 */
public interface Pings extends Feature {

    /**
     * Send a ping request to the server.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server
     *
     *         <p>
     *         If the ping was successful, the CompletableFuture will complete
     *         successfully. The result is a {@link PingDetails} containing
     *         timing information.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     * @since 6.0
     */
    CompletableFuture<PingDetails> pingServer();

    /**
     * Sends a ping request to the server.
     * <P>
     * The {@code callback}'s {@link Pings.PingCallback#onPingResponse(Pings.PingDetails)
     * onPingResponse} method will be called with the response.
     *
     * @param callback the callback for the response to this ping operation
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    void pingServer(PingCallback callback);

    /**
     * Sends a ping request to the server.
     * <P>
     * The {@code callback}'s
     * {@link Pings.PingContextCallback#onPingResponse(Object, Pings.PingDetails)
     * onPingResponse} method will be called with the response.
     *
     * @param context the context object that will be passed to the callback
     *
     * @param callback the callback for the response to this ping operation
     *
     * @param <C> context object type
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    <C> void pingServer(C context, PingContextCallback<C> callback);

    /**
     * Callback interface for {@link Pings#pingServer(PingCallback)}.
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    interface PingCallback extends Callback {
        /**
         * Called with ping response from the server.
         *
         * @param details information relating to the ping
         */
        void onPingResponse(PingDetails details);

        /**
         * Default implementation of {@link PingCallback}.
         * <P>
         * This simply logs onPingResponse calls at 'debug' level. This method
         * may be overridden to perform some more specific action.
         */
        class Default
            extends Callback.Default
            implements PingCallback {

            private static final Logger LOG =
                getLogger(PingCallback.Default.class);

            @Override
            public void onPingResponse(PingDetails details) {
                LOG.debug("{} - Ping response : {}", this, details);
            }

        }
    }

    /**
     * Callback interface for
     * {@link Pings#pingServer(Object, PingContextCallback)}.
     * <p>
     * Use this alternative to {@link PingCallback} to associate some arbitrary
     * context object with each call.
     *
     * @param <C> context object type
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    interface PingContextCallback<C> extends ContextCallback<C> {

        /**
         * Called with ping response from the server.
         *
         * @param context the context object supplied when making the call. May
         *        be {@code null}.
         *
         * @param details information relating to the ping
         */
        void onPingResponse(C context, PingDetails details);

        /**
         * Default implementation of {@link PingContextCallback}.
         * <P>
         * This simply logs onPingResponse calls at 'debug' level. This method
         * may be overridden to perform some more specific action.
         *
         * @param <C> context object type
         */
        class Default<C>
            extends ContextCallback.Default<C>
            implements PingContextCallback<C> {

            private static final Logger LOG =
                getLogger(PingContextCallback.Default.class);

            @Override
            public void onPingResponse(C context, PingDetails details) {
                LOG.debug(
                    "{} - Ping response {}, context={}",
                    this,
                    details,
                    context);
            }
        }
    }

    /**
     * Server response to a ping.
     */
    interface PingDetails {

        /**
         * Timestamp when the ping was sent.
         *
         * @return timestamp represented as milliseconds since epoch
         */
        long getTimestamp();

        /**
         * Round trip time in milliseconds from when the ping was sent to
         * the time the response was received.
         *
         * @return round trip time
         */
        long getRoundTripTime();
    }
}
