/*******************************************************************************
 * Copyright (c) 2014, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.features.control.clients;

import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import com.pushtechnology.diffusion.client.callbacks.Callback;
import com.pushtechnology.diffusion.client.callbacks.ContextCallback;
import com.pushtechnology.diffusion.client.session.PermissionsException;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.session.SessionClosedException;
import com.pushtechnology.diffusion.client.types.GlobalPermission;

/**
 * This feature allows a client session to query and update the system
 * authentication store.
 *
 * <H3>Access control</H3> In order to query the store the session needs
 * {@link GlobalPermission#VIEW_SECURITY VIEW_SECURITY} permission and in order
 * to update the store it needs {@link GlobalPermission#MODIFY_SECURITY
 * MODIFY_SECURITY} permission.
 *
 * <H3>Accessing the feature</H3> This feature may be obtained from a
 * {@link Session session} as follows:
 *
 * <pre>
 * SystemAuthenticationControl systemAuthenticationControl =
 *     session.feature(SystemAuthenticationControl.class);
 * </pre>
 *
 * @author DiffusionData Limited
 * @since 5.2
 */
public interface SystemAuthenticationControl extends SecurityStoreFeature {

    /**
     * Obtain the current contents of the store.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If the request was successful, the CompletableFuture will
     *         complete successfully with a
     *         {@link SystemAuthenticationConfiguration} result.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *
     *         <li>{@link PermissionsException} &ndash; if the session does
     *         not have {@code VIEW_SECURITY} permission;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     *
     * @since 6.0
     */
    CompletableFuture<SystemAuthenticationConfiguration>
        getSystemAuthentication();

    /**
     * Obtain the current contents of the store.
     *
     * @param callback the operation callback
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    void getSystemAuthentication(ConfigurationCallback callback);

    /**
     * Query the store for all of the system principals, with a contextual
     * callback.
     *
     * @param context the context to pass to the callback, may be null
     *
     * @param callback the operation callback
     *
     * @param <C> the context type
     *
     * @see #getSystemAuthentication(ConfigurationCallback)
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    <C> void getSystemAuthentication(
        C context,
        ConfigurationContextCallback<C> callback);

    /**
     * The callback interface for
     * {@link #getSystemAuthentication(ConfigurationCallback)
     * getSystemAuthentication}.
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    interface ConfigurationCallback extends Callback {
        /**
         * This is called to return the requested system authentication
         * configuration.
         *
         * @param configuration snapshot of information from the system
         *        authentication store
         */
        void onReply(SystemAuthenticationConfiguration configuration);
    }

    /**
     * The contextual callback interface for {@link ConfigurationCallback} for
     * {@link #getSystemAuthentication(Object, ConfigurationContextCallback)
     * getSystemAuthentication} .
     *
     * <p>
     * Attaches an arbitrary context object to callback notifications.
     *
     * @param <C> the context type
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    interface ConfigurationContextCallback<C> extends ContextCallback<C> {
        /**
         * This is called to return the requested system authentication
         * configuration.
         *
         * @param configuration snapshot of information from the system
         *        authentication store
         *
         * @param context the context object supplied when making the call
         */
        void onReply(
            C context,
            SystemAuthenticationConfiguration configuration);
    }

    /**
     * Action to be taken by the system authentication handler for connection
     * attempts that do not provide a principal name and credentials.
     */
    enum AnonymousConnectionAction {

        /**
         * Accept anonymous connection attempts.
         */
        ALLOW,
        /**
         * Deny anonymous connection attempts.
         */
        DENY,
        /**
         * Defer authentication decision for anonymous connection attempts to
         * subsequent authentication handlers.
         */
        ABSTAIN;
    }

    /**
     * Snapshot of information from the system authentication store.
     *
     * @see SystemAuthenticationControl#getSystemAuthentication(ConfigurationCallback)
     * @see SystemAuthenticationControl#getSystemAuthentication(Object,
     *      ConfigurationContextCallback)
     */
    interface SystemAuthenticationConfiguration {
        /**
         * Returns the system principals stored on the server.
         *
         * @return system principals
         */
        Collection<SystemPrincipal> getPrincipals();

        /**
         * Returns the action to take for anonymous connection attempts.
         *
         * @return action to take for anonymous connection attempts
         */
        AnonymousConnectionAction getAnonymousAction();

        /**
         * Returns the roles the system authentication handler will assign to
         * anonymous sessions. Applicable only if anonymous connections are
         * {@link AnonymousConnectionAction#ALLOW allowed}.
         *
         * @return roles that the system authentication handler will assign to
         *         anonymous sessions
         *
         * @see #getAnonymousAction()
         */
        Set<String> getRolesForAnonymousSessions();

        /**
         * Returns the map of trusted client proposed properties, where the key
         * is the permitted property name and the value defines the validation
         * applied to the property.
         *
         * @return map of trusted properties and their validation
         * @since 6.5
         */
        Map<String, SessionPropertyValidation>
            getTrustedClientProposedProperties();
    }

    /**
     * A principal in the system authentication store.
     */
    interface SystemPrincipal {

        /**
         * Returns the system principal name.
         *
         * @return principal name
         */
        String getName();

        /**
         * Returns the roles that the system authentication handler will assign
         * to the principal.
         *
         * @return roles
         */
        Set<String> getAssignedRoles();

        /**
         * Returns the name of the principal this principal is locked by.
         *
         * @return the name of the locking principal or an empty
         *         {@code Optional} if the this principal is not locked
         *
         * @see ScriptBuilder#addPrincipal(String, String, Set, String)
         * @since 6.4
         */
        Optional<String> getLockingPrincipal();
    }

    /**
     * Defines the validation for a trusted client proposed session property.
     *
     * <p>
     * The subtypes define specific types of validation.
     * </p>
     *
     * @since 6.5
     */
    interface SessionPropertyValidation {

        /**
         * Defines values based validation for a trusted client proposed session
         * property.
         *
         * <p>
         * Such validation defines a set of values to which the supplied session
         * property value must belong.
         * </p>
         *
         * @since 6.5
         */
        interface ValuesSessionPropertyValidation
            extends SessionPropertyValidation {

            /**
             * Returns the set of permitted values.
             *
             * @return set of permitted values
             */
            Set<String> getValues();
        }

        /**
         * Defines regular expression based validation for a trusted client
         * proposed session property.
         * <p>
         * Such validation defines a <a href=
         * "https://docs.oracle.com/javase/8/docs/api/java/util/regex/Pattern.html#sum">
         * regular expression</a> which supplied values must match against.
         * </p>
         *
         * @since 6.5
         */
        interface MatchesSessionPropertyValidation
            extends SessionPropertyValidation {

            /**
             * Returns the regular expression.
             *
             * @return the regular expression
             */
            String getRegex();
        }
    }

    /**
     * Returns a builder that can be used to create scripts for use with
     * {@link SecurityStoreFeature#updateStore updateStore}.
     *
     * @return an initial builder that creates an empty script
     */
    ScriptBuilder scriptBuilder();

    /**
     * A script builder may be used to create a script of commands to apply to
     * the system authentication store at the server.
     * <P>
     * Each method call on the builder adds a line to the script and then the
     * script may be built using the {@link ScriptBuilder#script() script}
     * method which produces a String script which may be sent to the server
     * using {@link SecurityStoreFeature#updateStore updateStore}.
     * <P>
     * Such a builder may be created using the
     * {@link SystemAuthenticationControl#scriptBuilder() scriptBuilder} method.
     * <p>
     * From Diffusion 6.5, script builders are no longer immutable. Each builder
     * operation mutates this script builder and returns it.
     */
    interface ScriptBuilder {

        /**
         * Add a new principal.
         * <P>
         * The script will fail if the principal is already defined at the
         * server.
         *
         * @param name principal name
         *
         * @param password password
         *
         * @param roles assigned roles, may be empty
         *
         * @return this builder, modified to add the principal
         */
        ScriptBuilder addPrincipal(
            String name, String password, Set<String> roles);

        /**
         * Add a new locked principal.
         * <P>
         * A locked principal can only be edited by the principal defined in the
         * lock. The script will fail if the principal is already defined at the
         * server.
         *
         * @param name principal name
         *
         * @param password password
         *
         * @param roles assigned roles, may be empty
         *
         * @param lockingPrincipal the name of the principal that can edit this
         *        principal
         *
         * @return this builder, modified to add the principal
         */
        ScriptBuilder addPrincipal(
            String name, String password, Set<String> roles,
            String lockingPrincipal);

        /**
         * Set a principal's password.
         * <P>
         * The principal must already be defined at the server in order to set
         * the password.
         *
         * @param principal principal name
         *
         * @param password password
         *
         * @return this builder, modified to set the password
         */
        ScriptBuilder setPassword(String principal, String password);

        /**
         * Assert that a principal's password is {@code password}.
         * <P>
         * This command does not update the store. It can be used in conjunction
         * with {@link #setPassword(String, String) setPassword} to create a
         * script that updates a password only if the previous password is
         * supplied.
         *
         * @param principal principal name
         *
         * @param password password
         *
         * @return this builder, modified to verify the password
         */
        ScriptBuilder verifyPassword(String principal, String password);

        /**
         * Change a principal's assigned roles.
         * <P>
         * The specified principal must already be defined at the server.
         *
         * @param principal principal name
         *
         * @param roles assigned roles
         *
         * @return this builder, modified to assign the roles
         */
        ScriptBuilder assignRoles(String principal, Set<String> roles);

        /**
         * Remove a principal.
         * <P>
         * The principal must be one that is already defined at the server.
         *
         * @param principal principal name
         *
         * @return this builder, modified to remove the principal
         */
        ScriptBuilder removePrincipal(String principal);

        /**
         * Instruct the system authentication handler to allow anonymous
         * connections.
         *
         * @param roles roles to assign to anonymous sessions. This may be empty
         *
         * @return this builder, modified to allow anonymous connections
         */
        ScriptBuilder allowAnonymousConnections(Set<String> roles);

        /**
         * Instruct the system authentication handler to deny anonymous
         * connections.
         *
         * @return this builder, modified to deny anonymous connections
         */
        ScriptBuilder denyAnonymousConnections();

        /**
         * Instruct the system authentication handler to defer authentication
         * decisions for anonymous connections to subsequent handlers.
         *
         * @return this builder, modified to abstain from authentication
         *         decisions for anonymous connections
         */
        ScriptBuilder abstainAnonymousConnections();

        /**
         * Specifies the name of a client proposed session property that should
         * be allowed by the system authenticator along with a set of
         * permissible values. The property will only be allowed if the supplied
         * value matches one of those in the set of values specified.
         *
         * @param propertyName specifies the name of the client proposed
         *        property to be allowed
         * @param allowedValues specifies a set of allowed values for the client
         *        proposed property
         *
         * @return this builder, modified to allow the specified client proposed
         *         property if its value matches one of those supplied
         *
         * @since 6.5
         */
        ScriptBuilder trustClientProposedPropertyIn(String propertyName,
            Set<String> allowedValues);

        /**
         * Specifies the name of a client proposed session property that should
         * be allowed by the system authenticator along with a regular
         * expression to validate the property value. The property will only be
         * allowed if the supplied value matches with the regular expression.
         *
         * @param propertyName specifies the name of the client proposed
         *        property to be allowed
         * @param regex <a href=
         *        "https://docs.oracle.com/javase/8/docs/api/java/util/regex/Pattern.html#sum">
         *        regular expression</a> which will be matched against supplied
         *        values to determine whether they are valid
         *
         * @return this builder, modified to allow the specified client proposed
         *         property if its value matches the given regular expression
         *
         * @since 6.5
         */
        ScriptBuilder trustClientProposedPropertyMatches(String propertyName,
            String regex);

        /**
         * Specifies the name of a client proposed session property that should
         * now be ignored by the system authenticator.
         * <p>
         * This removes the effect of a previous request to trust the named
         * property using {@link #trustClientProposedPropertyIn} or
         * {@link #trustClientProposedPropertyMatches}.
         *
         * @param propertyName specifies the name of the client proposed
         *        property to be ignored
         *
         * @return this builder, modified to ignore the specified client
         *         proposed property
         *
         * @since 6.5
         */
        ScriptBuilder ignoreClientProposedProperty(String propertyName);

        /**
         * Append all the operations of {@code other} to this ScriptBuilder.
         *
         * @return a combined script builder
         * @since 6.0
         */
        ScriptBuilder append(ScriptBuilder other);

        /**
         * Create a script.
         *
         * @return the script
         */
        String script();
    }
}
