/*******************************************************************************
 * Copyright (c) 2014, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.session.proxy;

import java.util.Map;

import com.pushtechnology.diffusion.client.session.SessionAttributes;

/**
 * Interface that defines a proxy authentication scheme.
 * <P>
 * The proxy authentication scheme should provide a {@link ChallengeHandler
 * challenge handler} which is responsible for reading challenge requests from
 * the proxy and providing the relevant response.
 *
 * @author DiffusionData Limited
 * @since 5.1
 * @see ProxyAuthenticationFactory
 */
public interface HTTPProxyAuthentication {

    /**
     * Allows for an authentication implementation to be provided for
     * connections routed via HTTP proxies.
     * <P>
     * The ChallengeHandler should read HTTP headers from a 401 or 407 challenge
     * and formulate a response.
     */
    public interface ChallengeHandler {

        /**
         * Called when a challenge from the proxy is received. The authenticator
         * will return a response to the challenge.
         *
         * @param challenge a map of the HTTP headers corresponding to the
         *        challenge received from the proxy
         *
         * @return a map of HTTP headers in response to the challenge
         */
        Map<String, String> getResponse(Map<String, String> challenge);

        /**
         * This method returns the maximum number of HTTP transactions permitted
         * by the authentication method before authentication fails.
         *
         * @return the maximum number of HTTP transactions
         */
        int getMaximumConversations();
    }

    /**
     * Create a challenge handler using this proxy authentication scheme. The
     * challenge handler is responsible for responding to authentication
     * challenges from the proxy during connection negotiation.
     *
     * @param sessionAttributes the session attributes
     *
     * @return a challenge handler for this authentication scheme
     */
    ChallengeHandler createHandler(SessionAttributes sessionAttributes);
}