/*******************************************************************************
 * Copyright (c) 2020, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.types;

import com.pushtechnology.diffusion.client.features.Messaging;
import com.pushtechnology.diffusion.client.features.TimeSeries;
import com.pushtechnology.diffusion.client.features.TimeSeries.Query;
import com.pushtechnology.diffusion.client.features.control.topics.SessionTrees;
import com.pushtechnology.diffusion.client.session.Session;

/**
 * Permissions protecting access-controlled operations that are evaluated for a
 * specific path.
 *
 * <p>
 * The meaning of the path depends on the permission. Most permissions apply to
 * topic paths in the topic tree. The {@link #SEND_TO_MESSAGE_HANDLER} and
 * {@link #SEND_TO_SESSION} permissions apply to {@link Messaging message
 * paths}. The {@link #ACQUIRE_LOCK} permission applies to
 * {@link Session#lock(String) lock names}.
 *
 * @author DiffusionData Limited
 * @since 6.5
 * @see GlobalPermission
 */
public enum PathPermission implements Permission {

    /**
     * Use a topic selector that selects a topic path.
     *
     * <p>
     * A session must have this permission for the
     * {@link com.pushtechnology.diffusion.client.topics.TopicSelector#getPathPrefix()
     * path prefix} of any topic selector used to subscribe or fetch.
     *
     * <p>
     * When the subscription or fetch request completes, the resulting topics
     * are further filtered based on the {@link #READ_TOPIC} permission.
     *
     * <p>
     * A session that has {@code READ_TOPIC} but not {@code SELECT_TOPIC} for a
     * particular topic path cannot subscribe directly to topics belonging to
     * the path. However, the session can be independently subscribed by a
     * control session that has {@link GlobalPermission#MODIFY_SESSION}
     * permission in addition to the appropriate {@code SELECT_TOPIC}
     * permission.
     *
     * <p>
     * A session granted {@code SELECT_TOPIC} for a particular path effectively
     * has the permission for all descendant paths. From a security perspective,
     * if a role grants {@code SELECT_TOPIC} at branch {@code X} it is
     * ineffectual for it also to deny {@code SELECT_TOPIC} at a child branch
     * {@code X/Y} because a a topic selector with a path prefix of {@code X}
     * can still select paths below {@code X/Y}.
     *
     * @since 5.7
     */
    SELECT_TOPIC,

    /**
     * Required to receive information from a topic.
     *
     * <p>
     * If a session does not have read_topic permission for a topic, the topic
     * will be excluded from the results of subscription or fetch operations for
     * the session, and the topic's details cannot be retrieved by the session.
     *
     * @see #SELECT_TOPIC
     */
    READ_TOPIC,

    /**
     * Update a topic.
     */
    UPDATE_TOPIC,

    /**
     * Add a topic or remove a topic.
     */
    MODIFY_TOPIC,

    /**
     * Send a message to a handler registered with the server for a particular
     * message path.
     */
    SEND_TO_MESSAGE_HANDLER,

    /**
     * Send a message to a client session for a particular message path.
     */
    SEND_TO_SESSION,

    /**
     * Evaluate queries that return a non-current view of a time series topic.
     *
     * <p>
     * The {@link #READ_TOPIC} permission is required to evaluate any type of
     * {@link Query} for a time series topic. This permission is additionally
     * required for queries that potentially return a non-current view of all or
     * part of a time series. Such queries include value range queries that
     * specify an edit range, and all types of edit range query.
     *
     * @see TimeSeries
     */
    QUERY_OBSOLETE_TIME_SERIES_EVENTS,

    /**
     * Submit edits to time series topic events.
     *
     * <p>
     * The {@link #UPDATE_TOPIC} permission is required to update a time series
     * topic. This permission is additionally required to submit
     * {@link TimeSeries#edit edits} to a time series topic.
     *
     * @see TimeSeries
     * @see #EDIT_OWN_TIME_SERIES_EVENTS
     */
    EDIT_TIME_SERIES_EVENTS,

    /**
     * Submit edits to time series topic events which have an author which is
     * the same as the principal of the calling session.
     *
     * <p>This permission is a more restrictive alternative to
     * {@link #EDIT_TIME_SERIES_EVENTS}.
     *
     * <p>
     * The {@link #UPDATE_TOPIC} permission is required to update a time series
     * topic. This permission is additionally required to submit
     * {@link TimeSeries#edit edits} to a time series topic where the event
     * author is the same as the principal of the calling session.
     *
     * @see TimeSeries
     * @see #EDIT_TIME_SERIES_EVENTS
     */
    EDIT_OWN_TIME_SERIES_EVENTS,

    /**
     * Acquire a session lock.
     *
     * @see Session#lock
     */
    ACQUIRE_LOCK,

    /**
     * Expose a branch of the topic tree as a virtual session tree.
     *
     * <p>
     * The {@code EXPOSE_BRANCH} path permission is powerful since it allows a
     * session to expose a whole branch of the topic tree under a different set
     * of path permissions.
     *
     * <p>
     * A session granted {@code EXPOSE_BRANCH} for a particular path effectively
     * has the permission for all descendant paths. From a security perspective,
     * if a role grants {@code EXPOSE_BRANCH} at branch {@code X} it is
     * ineffectual for it also to deny {@code EXPOSE_BRANCH} at a child branch
     * {@code X/Y} because a branch mapping to {@code X} can still expose paths
     * below {@code X/Y}.
     *
     * @see SessionTrees
     * @since 6.7
     */
    EXPOSE_BRANCH,

    /**
     * A permission that is unsupported by the session.
     */
    UNKNOWN_PATH_PERMISSION,
}
