/*******************************************************************************
 * Copyright (c) 2016, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype.json;

import com.pushtechnology.diffusion.datatype.BinaryDelta;
import com.pushtechnology.diffusion.datatype.Bytes;
import com.pushtechnology.diffusion.datatype.DataType;
import com.pushtechnology.diffusion.datatype.InvalidDataException;

/**
 * Immutable JSON data value.
 *
 * <p>
 * JSON is "JavaScript Object Notation", a lightweight data-interchange format.
 * See <a href="http://www.json.org">www.json.org</a>.
 *
 * <p>
 * To create an instance from a JSON string, obtain a {@code JSONDataType}
 * implementation from
 * {@link com.pushtechnology.diffusion.datatype.DataTypes#json()
 * Diffusion.dataTypes().json()} and call
 * {@link JSONDataType#fromJsonString(String)}.
 *
 * <p>
 * The {@link #toJsonString()} method can be used to retrieve this value as a
 * JSON string. Applications can use a JSON library such as <a
 * href="http://wiki.fasterxml.com/JacksonHome">Jackson</a> to create and
 * interpret the JSON data represented as a string.
 *
 * <h2>CBOR representation</h2>
 *
 * <p>
 * Internally the value is stored and transmitted not as a JSON string, but in
 * CBOR format to reduce memory and network overhead. CBOR (Concise Binary
 * Object Representation) is a standardized format for binary representation of
 * structured data defined by <a href= "https://tools.ietf.org/html/rfc7049">RFC
 * 7049</a>. See <a href="http://www.cbor.io">www.cbor.io</a>.
 *
 * <p>
 * Rather than working with JSON strings it is possible, and usually preferable,
 * for applications to work directly with the underlying CBOR-format binary
 * representation. This avoids creating intermediate JSON strings, and allows
 * access to CBOR features such as the byte string data type.
 *
 * <p>
 * Each JSON value is represented as a single CBOR data item. CBOR supports
 * composite data types just like JSON, so this data item can be an array or a
 * map of other data items. The JSON {@code null} value is represented by the
 * CBOR {@code Null} value.
 *
 * <p>
 * A particular advantage of working directly with the CBOR-format data is that
 * binary data can be written to the value as a data item using the CBOR byte
 * string type. The data item will be stored as part of the JSON value in binary
 * form and transmitted without further conversion.
 *
 * <h2>Working with CBOR data</h2>
 *
 * <p>
 * We recommend Jackson's <a
 * href="https://github.com/FasterXML/jackson-dataformat-cbor">jackson-
 * dataformat-cbor</a> as a mature library for generating and parsing
 * CBOR-format binary data.
 *
 * <p>
 * To create an instance from a CBOR format byte array, call
 * {@link JSONDataType#readValue(byte[])}. The byte array must contain a single
 * CBOR data item, otherwise, an {@code InvalidDataException} will be thrown
 * when the data is first parsed. To convert an instance to CBOR format, use
 * {@link #asInputStream()} or
 * {@link com.pushtechnology.diffusion.datatype.DataType#writeValue(Object, java.io.OutputStream)
 * JSONDataType.writeValue(JSON, OutputStream)}.
 *
 * <p>
 * Here is an example of using {@code jackson-dataformat-cbor} to parse a CBOR
 * map of text values.
 *
 * <pre>
 * Map&lt;String, String&gt; parseMap(JSON value) throws IOException {
 *     CBORFactory factory = new CBORFactory();
 *     CBORParser parser = factory.createParser(value.asInputStream());
 *
 *     if (parser.nextToken() != JsonToken.START_OBJECT) {
 *         return Collections.emptyMap();
 *     }
 *
 *     Map&lt;String, String&gt; result = new HashMap&lt;&gt;();
 *
 *     while (true) {
 *         String key = parser.nextFieldName();
 *
 *         if (key == null) {
 *             return result;
 *         }
 *
 *         result.put(key, parser.nextTextValue());
 *     }
 * }
 * </pre>
 *
 * <p>
 * Parsing is straightforward for simple structures, but with more complex
 * structures it can become unwieldy. For a higher-level interface, a binding
 * library such as <a
 * href="https://github.com/FasterXML/jackson-databind">jackson-databind</a> can
 * be used to map CBOR data to and from instances of application Java classes.
 * Here is an example of using {@code jackson-dataformat-cbor} and
 * {@code jackson-databind} to parse a JSON object of text values from the CBOR
 * binary data.
 *
 * <pre>
 * // mapper and factory can be reused
 * private ObjectMapper mapper = new ObjectMapper();
 * private CBORFactory factory = new CBORFactory();
 *
 * Map&lt;String, String&gt; parseObject(JSON value) throws IOException {
 *
 *     CBORParser parser = factory.createParser(value.asInputStream());
 *
 *     return (Map&lt;String, String&gt;) mapper.readValue(parser, Map.class);
 * }
 * </pre>
 *
 * <h2>CBOR limitations</h2>
 *
 * <p>
 * The mapping from a JSON string to and from CBOR-format binary data is mostly
 * straightforward, but there are a few limitations. In particular:
 * <ul>
 * <li>Non-string keys are not supported.
 * <li>Precision may be lost converting numbers from JSON to CBOR. For example,
 * a JSON string can precisely represent the decimal fraction {@code 0.3} as
 * text. The default {@link JSONDataType#fromJsonString(String)} conversion will
 * convert this imprecisely as a floating-point number. Applications can work
 * around this limitation by reading and writing CBOR data directly to a decimal
 * fraction tagged array.
 * </ul>
 *
 * @author DiffusionData Limited
 * @since 5.7
 */
public interface JSON extends Bytes {

    /**
     * Compare this JSON value with an earlier version to create a binary delta.
     *
     * <p>
     * Convenient equivalent to
     * {@link com.pushtechnology.diffusion.datatype.DeltaType#diff(Object, Object)
     * Diffusion.dataTypes().json().binaryDeltaType().diff(original, this)}.
     *
     * @return a delta representing the difference between original and this
     *         JSON
     * @throws InvalidDataException if the original or this instance is invalid
     */
    BinaryDelta binaryDiff(JSON original) throws InvalidDataException;

    /**
     * Apply a binary delta to this JSON value to create a new value.
     *
     * <p>
     * Convenient equivalent to
     * {@link com.pushtechnology.diffusion.datatype.DeltaType#apply(Object, Object)
     * Diffusion.dataTypes().json().binaryDeltaType().apply(this, delta)}.
     *
     * @throws InvalidDataException if this instance or delta is invalid
     */
    JSON apply(BinaryDelta delta) throws InvalidDataException;

    /**
     * Compare this JSON value with an earlier version to calculate a structural
     * delta.
     *
     * <p>
     * Unlike a {@link #binaryDiff(JSON) binary delta}, a structural delta can
     * be queried to determine its effect.
     *
     * @param original the original JSON value to compare with this value
     * @return a structural delta representing the difference between original
     *         and this JSON
     * @throws InvalidDataException if original or this instance is invalid
     */
    JSONDelta diff(JSON original) throws InvalidDataException;

    /**
     * Check whether this instance is valid.
     *
     * <p>
     * Convenient equivalent to {@link DataType#validate(Object)
     * Diffusion.dataTypes().json().validate(this)}.
     *
     * @return This instance to allow fluent chaining. For example:
     *
     *         <pre>
     * JSON json = jsonDataType.read(data).validate()
     * </pre>
     *
     * @throws InvalidDataException if this value is invalid
     */
    JSON validate() throws InvalidDataException;

    /**
     * @return JSON format string
     *
     * @throws InvalidDataException if the underlying CBOR data is not valid;
     *         for example, if it contains an invalid CBOR sequence, no data, or
     *         more that one CBOR data item
     */
    String toJsonString() throws InvalidDataException;
}
