/*******************************************************************************
 * Copyright (c) 2017, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype.recordv2;

import java.util.List;

import com.pushtechnology.diffusion.datatype.Bytes;
import com.pushtechnology.diffusion.datatype.InvalidDataException;
import com.pushtechnology.diffusion.datatype.recordv2.model.RecordModel;
import com.pushtechnology.diffusion.datatype.recordv2.schema.Schema;

/**
 * An immutable value representing a list of records.
 *
 * @author DiffusionData Limited
 * @since 6.0
 * @see RecordV2DataType
 */
public interface RecordV2 extends Bytes {

    /**
     * Compare this value with an earlier version to calculate a structural
     * delta.
     *
     * @param original the original value to compare with this value
     *
     * @return a structural delta representing the difference between the
     *         original and this value
     */
    RecordV2Delta diff(RecordV2 original);

    /**
     * Parses the value into a model based upon a specified schema.
     * <P>
     * This assumes that data is compatible with the schema and does not do any
     * validation. There is no need to validate the data if this has been done
     * on entry or at the server. However, if the data is invalid then issues
     * may occur when attempting to access it.
     * <P>
     * If it is not certain that the data is valid then the
     * {@link #asValidatedModel(Schema)} method may be used instead.
     *
     * @param schema the schema to use for parsing the data
     *
     * @return an immutable model derived from the data value
     */
    RecordModel asModel(Schema schema);

    /**
     * Parses the value into a model based upon a specified schema.
     *
     * @param schema the schema to use for parsing the data
     *
     * @return a model derived from the value
     *
     * @throws InvalidDataException if the data value is incompatible with the
     *         supplied schema
     */
    RecordModel asValidatedModel(Schema schema)
        throws InvalidDataException;

    /**
     * Returns the data value as a list of lists of strings.
     *
     * @return new mutable list where each entry represents a record within the
     *         data
     */
    List<List<String>> asRecords();

    /**
     * Returns the data value as a list of fields.
     * <P>
     * This disregards record boundaries. If there is more than one record, they
     * are concatenated to produce a list of all of the fields.
     * <P>
     * This method would normally only be used when it is known that there is
     * only one record.
     *
     * @return new mutable list of all of the fields
     */
    List<String> asFields();

}
