/*******************************************************************************
 * Copyright (c) 2014, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.types;

/**
 * Immutable value containing the proof material to support an authentication
 * request.
 * <P>
 * The server passes the credentials through to the configured authentication
 * handlers for interpretation.
 * <P>
 * Credentials can be created using a {@link Credentials.Factory}, an instance
 * of which can be obtained as follows:
 *
 * <pre>
 * <code>
 * Credentials.Factory factory = Diffusion.credentials();
 * </code>
 * </pre>
 *
 * @author DiffusionData Limited
 * @since 5.0
 */
public interface Credentials {

    /**
     * Return the type of credentials.
     *
     * @return The type of the credentials
     */
    Type getType();

    /**
     * Return the credential data as a byte array.
     *
     * @return the data
     */
    byte[] toBytes();

    /**
     * Type code indicated the type of the credentials.
     */
    enum Type {

        /**
         * No credentials are supplied. This is used for anonymous session
         * creation. It may also be supported by authentication handlers that
         * allow identity to be changed without further credentials.
         */
        NONE,

        /**
         * A plain-text password, encoded with UTF-8.
         */
        PLAIN_PASSWORD,

        /** Application-specific credentials. */
        CUSTOM;
    }

    /**
     * Factory for the standard credential types.
     * <P>
     * An instance of this factory can be obtained as follows:
     *
     * <pre>
     * <code>
     * Credentials.Factory factory = Diffusion.credentials();
     * </code>
     * </pre>
     */
    interface Factory {
        /**
         * Create a {@link Credentials} of type {@link Type#NONE NONE}.
         *
         * @return the credentials
         */
        Credentials noCredentials();

        /**
         * Create a {@link Credentials} of type {@link Type#PLAIN_PASSWORD
         * PLAIN_PASSWORD}.
         *
         * @param password the password
         * @return the credentials
         */
        Credentials password(String password);

        /**
         * Create a {@link Credentials} of type {@link Type#CUSTOM CUSTOM}.
         *
         * @return the credentials
         */
        Credentials custom(byte[] bytes);
    }

}
