/*******************************************************************************
 * Copyright (c) 2016, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype;

import com.pushtechnology.diffusion.datatype.binary.BinaryDataType;
import com.pushtechnology.diffusion.datatype.json.JSON;
import com.pushtechnology.diffusion.datatype.json.JSONDataType;
import com.pushtechnology.diffusion.datatype.recordv2.RecordV2DataType;

/**
 * Diffusion data type implementations.
 *
 * <p>
 * An instance can be obtained by calling
 * {@link com.pushtechnology.diffusion.client.Diffusion#dataTypes()
 * Diffusion.dataTypes()}.
 *
 * @author DiffusionData Limited
 * @since 5.7
 */
public interface DataTypes {

    /**
     * Binary data type name.
     *
     * @since 6.0
     * @see #getByName(String)
     * @see #binary()
     */
    String BINARY_DATATYPE_NAME = "binary";

    /**
     * Double data type name.
     *
     * @since 6.0
     * @see #getByName(String)
     * @see #doubleFloat()
     */
    String DOUBLE_DATATYPE_NAME = "double";

    /**
     * INT64 data type name.
     *
     * @since 6.0
     * @see #getByName(String)
     * @see #int64()
     */
    String INT64_DATATYPE_NAME = "int64";

    /**
     * JSON data type name.
     *
     * @since 6.0
     * @see #getByName(String)
     * @see #json()
     */
    String JSON_DATATYPE_NAME = "json";

    /**
     * String data type name.
     *
     * @since 6.0
     * @see #getByName(String)
     * @see #string()
     */
    String STRING_DATATYPE_NAME = "string";

    /**
     * RecordV2 data type name.
     *
     * @since 6.0
     * @see #getByName(String)
     * @see #recordV2()
     */
    String RECORD_V2_DATATYPE_NAME = "record_v2";

    /**
     * JSON is "JavaScript Object Notation", a lightweight data-interchange
     * format. See <a href="http://www.json.org">www.json.org</a>.
     *
     * @return JSON data type
     */
    JSONDataType json();

    /**
     * Data type that supports arbitrary binary data.
     *
     * @return Binary data type
     */
    BinaryDataType binary();

    /**
     * Data type that supports data in the form of records and fields.
     *
     * @return RecordV2 data type
     */
    RecordV2DataType recordV2();

    /**
     * Data type that supports double-precision floating point numbers.
     * (Eight-byte IEEE 754).
     *
     * <p>
     * The integer value is serialized as CBOR-format binary. A serialized value
     * can be {@link DataType#readAs(Class, Bytes) read as} a {@link JSON}
     * instance.
     *
     * <p>
     * The data type does not support binary deltas.
     * <p>
     * The data type supports null Double instances.
     *
     * @return floating point data type
     * @since 6.0
     */
    DataType<Double> doubleFloat();

    /**
     * Data type that supports 64-bit, signed integer values.
     *
     * <p>
     * The integer value is serialized as CBOR-format binary. A serialized value
     * can be {@link DataType#readAs(Class, Bytes) read as} a {@link JSON}
     * instance.
     *
     * <p>
     * The data type does not support binary deltas.
     * <p>
     * The data type supports null Long instances.
     *
     * @return integer data type
     * @since 6.0
     */
    DataType<Long> int64();

    /**
     * Data type that supports string values.
     *
     * <p>
     * The string value is serialized as CBOR-format binary. A serialized value
     * can be {@link DataType#readAs(Class, Bytes) read as} a {@link JSON}
     * instance.
     *
     * <p>
     * The data type supports binary deltas.
     * <p>
     * The data type supports null String instances.
     *
     * @return String data type
     * @since 6.0
     */
    DataType<String> string();

    /**
     * Obtain a {@link DataType} implementation by type name.
     *
     * @param typeName the name, as returned by {@link DataType#getTypeName()}
     * @return the data type
     * @throws IllegalArgumentException if there is no data type with the
     *         name {@code typeName}
     */
    DataType<?> getByName(String typeName);

    /**
     * Check if a given {@code valueClass} is a valid {@link DataType}
     * or a superclass of one.
     *
     * @param valueClass the class
     * @throws IllegalArgumentException if there is no data type that supports
     *         values of class {@code valueClass}.
     */
    void validateValueClass(Class<?> valueClass);

    /**
     * Obtain a {@link DataType} implementation by class.
     *
     * @param valueClass the class
     * @return the data type
     * @throws IllegalArgumentException if there is no data type that supports
     *         values of class {@code valueClass}
     * @throws IllegalArgumentException if there is more than one data type that
     *         supports values of the given class, but none is preferred
     */
    <V> DataType<V> getByClass(Class<V> valueClass);
}
