/*******************************************************************************
 * Copyright (c) 2017, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype.recordv2;

import java.util.List;

import com.pushtechnology.diffusion.datatype.recordv2.schema.Schema;

/**
 * {@link RecordV2} structural delta.
 * <p>
 * A RecordV2Delta describes the differences between two {@link RecordV2}
 * values. Unlike a {@link com.pushtechnology.diffusion.datatype.BinaryDelta
 * binary delta}, a structural delta can be queried to determine its effect. The
 * {@link #changes} method provides details of which values have changed.
 *
 * <p>
 * An instance can be created from two RecordV2 values using
 * {@link RecordV2#diff(RecordV2)}.
 *
 * <p>
 * RecordV2Deltas are useful for identifying small changes to complex RecordV2
 * values. Here's any example of how to use this class to filter interesting
 * changes in a
 * {@link com.pushtechnology.diffusion.client.features.Topics.ValueStream
 * ValueStream}.
 *
 * <pre>
 * public class ExampleStream implements ValueStream&lt;RecordV2&gt; {
 *   public void onValue(String topicPath, RecordV2 oldValue, RecordV2 newValue) {
 *     RecordV2Delta delta = newValue.diff(oldValue);
 *
 *     for (Change change : delta.changes(schema)) {
 *         if (change.fieldName().equals("address") {
 *             processAddress(newValue);
 *         }
 *     }
 *   }
 *
 *   // ...
 * }
 * </pre>
 *
 * @author DiffusionData Limited
 * @since 6.0
 */
public interface RecordV2Delta {

    /**
     * Returns a list of the changes represented by the delta with reference to
     * a specified schema.
     * <P>
     * The schema supplied must comply with the data format of the delta. No
     * validation takes place, so if the schema does not match the data then the
     * results may be unpredictable.
     *
     * @param schema the schema
     * @return the list of changes
     */
    List<Change> changes(Schema schema);

    /**
     * Represents a single change between one record value and another.
     */
    interface Change {

        /**
         * Indicates the type of change.
         */
        enum Type {
            /**
             * The change indicates that a field had its value changed.
             * <P>
             * This could be a field that has had its value changed or a new
             * field that has been added at the end of a variable length record.
             * <P>
             * The change will contain name and index details of both the record
             * and the field.
             */
            FIELD_CHANGED,
            /**
             * The change indicates that one or more field values have been
             * added.
             * <P>
             * This will only occur when variable multiplicity fields are used
             * within records.
             * <P>
             * The change will contain name and index of the record and the name
             * and index of the first field added.
             */
            FIELDS_ADDED,
            /**
             * The change indicates that a field value has been removed.
             * <P>
             * This will only occur when variable multiplicity fields are used
             * within records.
             * <P>
             * The change will contain name and index details of the record and
             * the name and index of the first field removed.
             */
            FIELDS_REMOVED,
            /**
             * The change indicates that one or more records have been added.
             * <P>
             * This will only occur when variable multiplicity records are in
             * use.
             * <P>
             * The change will contain only the record name and the index of the
             * first record added.
             */
            RECORDS_ADDED,
            /**
             * The change indicates that one or more records have been removed.
             * <P>
             * This will only occur when variable multiplicity records are in
             * use.
             * <P>
             * The change will contain only the record name and the index of the
             * first record removed.
             */
            RECORDS_REMOVED
        }

        /**
         * Returns the change type.
         *
         * @return change type.
         */
        Type type();

        /**
         * Returns the name of the affected record.
         *
         * @return record name
         */
        String recordName();

        /**
         * Returns the index of the affected record.
         * <P>
         * The index of the first (or only) record occurrence with the given
         * name will be 0.
         *
         * @return record index
         */
        int recordIndex();

        /**
         * Returns the name of the affected field.
         * <P>
         * If type is {@link Type#RECORDS_ADDED RECORDS_ADDED} or
         * {@link Type#RECORDS_REMOVED RECORDS_REMOVED} then a zero length
         * string will be returned.
         *
         * @return field name
         */
        String fieldName();

        /**
         * Returns the index of the affected field.
         * <P>
         * If type is {@link Type#RECORDS_ADDED RECORDS_ADDED} or
         * {@link Type#RECORDS_REMOVED RECORDS_REMOVED} then zero will be
         * returned.
         *
         * @return field index
         */
        int fieldIndex();

        /**
         * Returns the string key representation of the affected item in the
         * form recordName(recordIndex).fieldName(fieldIndex) or just
         * recordName(recordIndex) in the case of {@link Type#RECORDS_ADDED
         * RECORDS_ADDED} or {@link Type#RECORDS_REMOVED RECORDS_REMOVED}.
         *
         * @return the key of the affected item
         */
        String key();
    }

}
