/*******************************************************************************
 * Copyright (c) 2017, 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype.recordv2.schema;

import java.util.List;

import com.pushtechnology.diffusion.datatype.recordv2.RecordV2;
import com.pushtechnology.diffusion.datatype.recordv2.model.MutableRecordModel;

/**
 * {@link RecordV2} schema.
 * <P>
 * A schema describes data value format in terms of one or more record definitions.
 * A record definition describes the layout of a record and comprises one or more
 * field definitions.
 * <P>
 * Within the data value there can be multiple occurrences of a record or
 * field described by a single definition. The defined (or allowed, when
 * describing variable numbers) number of occurrences of each definition is
 * referred to as its 'multiplicity'. The multiplicity can be fixed (the item
 * occurs a fixed number of times), or variable (the item occurs from a minimum
 * number of times to a maximum number of times). If a variable field is used it
 * must be the last in a record definition and if a variable record is used it
 * must be the last in the schema definition.
 * <P>
 * A field may be defined as of type 'string', 'integer' or 'decimal'. A decimal
 * type has a further property of 'scale' which defines the number of digits to
 * the right of the decimal point.
 *
 * @author DiffusionData Limited
 * @since 6.0
 */
public interface Schema {

    /**
     * Returns an immutable, ordered list of record definitions.
     * <P>
     * There will be at least one.
     *
     * @return a list of the record definitions in the schema
     */
    List<Record> getRecords();

    /**
     * Create a mutable model based upon the schema.
     * <P>
     * The model will be created with all mandatory record occurrences and all
     * mandatory field occurrences initialized to default values.
     * <P>
     * Such a model may be mutated and used to generate updated
     * {@link RecordV2} occurrences for updating purposes.
     *
     * @return a new initialized model
     */
    MutableRecordModel createMutableModel();

    /**
     * Returns the schema in a JSON format.
     *
     * @return schema in JSON string format
     */
    String asJSONString();

    /**
     * Schema Node - A {@link Record} or a {@link Field}.
     */
    interface Node {
        /**
         * Returns the node name.
         *
         * @return the node name
         */
        String getName();

        /**
         * Returns minimum number of occurrences.
         * <P>
         * This may be 0 for a variable multiplicity field; otherwise, it will
         * be a positive value.
         *
         * @return the minimum number of occurrences of the node within its
         *         parent
         */
        int getMin();

        /**
         * Returns maximum number of occurrences.
         * <P>
         * This will be a positive value greater than or equal to the
         * {@link #getMin() min} value, or -1 to indicate an unlimited number.
         *
         * @return the maximum number of occurrences of the node within its
         *         parent
         */
        int getMax();

        /**
         * Returns true of the node has variable multiplicity.<BR>
         * i.e. min != max.
         *
         * @return true if the node has variable multiplicity, or false if it
         *         has fixed multiplicity
         */
        boolean isVariable();
    }

    /**
     * Record Definition.
     */
    interface Record extends Node {
        /**
         * Returns an immutable, ordered list of field definitions.
         * <P>
         * There will be at least one.
         *
         * @return a list of the field definitions within the record
         */
        List<Field> getFields();
    }

    /**
     * Field Definition.
     */
    interface Field extends Node {

        /**
         * Field Type.
         */
        enum Type {
            /**
             * Plain String.
             */
            STRING,
            /**
             * Integer.
             */
            INTEGER,
            /**
             * Decimal.
             */
            DECIMAL
        }

        /**
         * Returns the field type.
         *
         * @return the field type
         */
        Type getType();

        /**
         * Returns the scale of a {@link Type#DECIMAL decimal} field or 0 for
         * other types.
         * <P>
         * The scale represents the number of places to the right of the decimal
         * point in a decimal number.
         *
         * @return the scale of a decimal field
         */
        int getScale();

    }

}
